__doc__ = """SQL Blender

$Id: SQLBlender.py,v 1.1 2001/09/05 03:05:32 andy Exp $"""
__version__ = "$Id: SQLBlender.py,v 1.1 2001/09/05 03:05:32 andy Exp $"[11:-2]

import os
from Globals import HTMLFile, DTMLFile, InitializeClass
from OFS import PropertyManager, ObjectManager, PropertySheets
from AccessControl import ClassSecurityInfo
import SQLBColumn

manage_addSQLBlenderForm = HTMLFile(
    os.path.join('dtml','manage_addSQLBlenderForm'), globals())

def _get_connection(context):
    try:
        return getattr(context, context.connection_id)()
    except AttributeError:
        raise AttributeError, \
              "The database connection %s could not be found." % \
              context.connection_id

def spacify(s):
    from string import split, join
    return join(split(s,"_"))

def capwords(s):
    from string import split, join, capitalize
    return join(map(capitalize, split(s)))

def manage_addSQLBlender(context, id="sql_blender", title="",
                         connection_id=None, table=None, all_tables=None,
                         build_everything=None,
                         build_folder=None,
                         spacify_labels=None,
                         capitalize_labels=None,
                         capwords_labels=None,
                         REQUEST=None):
    """Add a SQL Blender to a Folder."""
    from string import join
    labeling = {}
    messages = []
    if spacify_labels: labeling['spacify'] = 1
    if capitalize_labels: labeling['capitalize'] = 1
    if capwords_labels: labeling['capwords'] = 1
    if all_tables:
        c = getattr(context, connection_id)()
        title_suffix = title
        for t in c.tables():
            if t['TABLE_TYPE'] == 'SYSTEM_TABLE': continue
            table = t['TABLE_NAME']
            b_id = "%s_sql_blender" % table
            if title_suffix:
                b_title = "%s %s" % (table, title_suffix)
            else:
                b_title = ''
            b = SQLBlender(b_id, b_title, connection_id, table, build_folder)
            b.__of__(context).build_default(labeling)
            if build_everything:
                b.__of__(context).manage_build(table)
            context._setObject(b.id, b)
            get_transaction().commit(1)
        if REQUEST is not None:
            context.manage_main(context, REQUEST)
    else:
        b = SQLBlender(id, title, connection_id, table, build_folder)
        b.__of__(context).build_default(labeling)
        if build_everything:
            messages.extend(b.__of__(context).manage_build(table))
        context._setObject(id, b)

    if REQUEST is not None:
        message = '<UL>\n%s</UL>' % join(messages,'\n  ')
        context.manage_main(context, REQUEST) #, manage_tabs_message=message)


class SQLBlender(
    PropertyManager.PropertyManager,
    ObjectManager.ObjectManager
    ):

    "SQL Blender"

    meta_type = "SQL Blender"
    icon = "misc_/SQLBlender/SQLBlenderIcon"
    #meta_types = ({'name': SQLBColumn.SQLBColumn.meta_type,
    #               'action': 'manage_addSQLBColumnForm'},)
    security = ClassSecurityInfo()
    security.declareProtected('View management screens',
                              'manage_buildForm', 'manage_build')

    manage_options = ObjectManager.ObjectManager.manage_options + \
                     PropertyManager.PropertyManager.manage_options + \
                     (
        {'label': 'Build', 'action': 'manage_buildForm'},
        )
    
    _properties = (
        {'id': 'connection_id', 'type': 'selection',
         'select_variable': 'SQLConnectionIDs', 'mode': 'w'},
        {'id': 'table', 'type': 'selection',
         'select_variable': 'ListTables', 'mode': 'w'},
        {'id': 'build_folder', 'type': 'string', 'mode': 'w'},
        {'id': 'column_order', 'type': 'tokens', 'mode': 'w'},
        {'id': 'FROM', 'type': 'string', 'mode': 'w'},
        {'id': 'WHERE', 'type': 'string', 'mode': 'w'},
        {'id': 'TR_even_odd', 'type': 'boolean', 'mode': 'w'},
        )
                      
    def __init__(self, id, title, connection_id, table, build_folder):
        self.propertysheets = PropertySheets.DefaultPropertySheets()
        self.id = id
        self.title = title
        self.connection_id = connection_id
        self.table = table
        self.build_folder = build_folder
        self.column_order = []
        self.FROM = ''
        self.WHERE = ''
        self.TR_even_odd = 0

    def build_default(self, labeling):
        from string import capitalize
        if not self.table: return
        columns = self.Columns(self.table)
        for column in columns:
            column_name = column['Name']
            NOT_NULL = not (column['Nullable'] and 1 or 0)
            if column_name == 'id':
                AS = column_id = "id_"
            else:
                column_id = column_name
                AS = ''
            if labeling:
                label = column_name
                if labeling.get('spacify'):
                    label = spacify(label)
                if labeling.get('capitalize'):
                    label = capitalize(label)
                if labeling.get('capwords'):
                    label = capwords(label)
            else:
                label = ''
            self.column_order.append(column_id)
            c = SQLBColumn.SQLBColumn(
                id=column_id,
                column=column_name,
                label=label,
                AS=AS,
                field_type=column['Type'],
                scale=column.get('Scale',0),
                precision=column.get('Precision',0),
                default=column.get('Default',''),
                PRIMARY_KEY=column.get("PrimaryKey",0),
                INDEX=column.get('Index',0),
                UNIQUE=column.get('Unique',0),
                NOT_NULL=NOT_NULL,
                )
            self._setObject(c.id, c)
        self._p_updated=1

    def manage_build(self, id='', title='', build_folder='', REQUEST=None):
        """Build a Folder containing the generated Z SQL Methods and
        DTML Methods."""
        from OFS.Folder import manage_addFolder
        from OFS.DTMLMethod import addDTMLMethod
        from Products.ZSQLMethods.SQL import manage_addZSQLMethod
        SQL = manage_addZSQLMethod
        from string import join
        messages = []
        parent = self.aq_parent
        id = id or self.table
        build_folder = build_folder or self.build_folder
        if build_folder:
            try:
                bf = getattr(parent, build_folder)
            except AttributeError:
                manage_addFolder(parent, build_folder, title)
                bf = getattr(parent, build_folder)
                messages.append("<LI> Folder %s created" % \
                                build_folder)
        else:
            bf = parent
        try:
            manage_addFolder(bf, id, title)
        except:
            messages.append('<LI> Folder %s exists, updating' % id)
        f = getattr(bf, id)
        SQL(f, 'SELECT', 'SELECT ... FROM %s' % self.table,
            self.connection_id, join(self.SELECT_args()),
            self.SELECT_stmt())
        SQL(f, 'INSERT', 'INSERT INTO %s ...' % self.table,
            self.connection_id, join(self.INSERT_args()),
            self.INSERT_stmt())
        UPDATE = self.UPDATE_stmt()
        if UPDATE:
            SQL(f, 'UPDATE', 'UPDATE %s ...' % self.table,
                self.connection_id, join(self.UPDATE_args()),
                self.UPDATE_stmt())
        addDTMLMethod(f, id='table_heading', file=self.table_heading_dtml())
        if self.TR_even_odd:
            addDTMLMethod(f, id='table_row_odd', file=self.table_row_dtml())
            addDTMLMethod(f, id='table_row_even', file=self.table_row_dtml(1))
        else:
            addDTMLMethod(f, id='table_row', file=self.table_row_dtml())
        addDTMLMethod(f, id='list', file=self.list_dtml())
        addDTMLMethod(f, id='form', file=self.form_dtml())
        addDTMLMethod(f, id='show', file=self.show_dtml())
        self.build_select_options_queries(f)
        if REQUEST is not None:
            messages.append("<LI> Content created in %s/%s." % \
                            (build_folder, id))
            message = '<UL>%s</UL>' % join(messages,'\n  ')
            return self.manage_buildForm(REQUEST,
                                         management_view="Build",
                                         manage_tabs_message=message)
        else:
            return messages
                          
    manage_buildForm = DTMLFile(
        os.path.join("dtml","manage_buildForm"), globals())

    def build_select_options_queries(self, folder):
        from OFS.DTMLMethod import addDTMLMethod
        from Products.ZSQLMethods.SQL import manage_addZSQLMethod
        SQL = manage_addZSQLMethod
        for column in self.column_list():
            args, query = self.select_options_query(column)
            if args and query:
                SQL(folder, "SELECT_%s_options" % column.column_reference(),
                   '', self.connection_id, args, query)
                addDTMLMethod(folder,
                              id='select_%s' % column.column_reference(),
                              file=self.select_dtml(column.column_reference(),
                                                    column.display_column))
                
    def append_new_column(self, column):
        self.column_order.append(column)
        self._p_changed = 1

    def get_connection(self): return _get_connection(self)
    
    def Tables(self):
        c = self.get_connection()
        tables = c.tables()
        for t in tables:
            t['Columns'] = c.columns(t['TABLE_NAME'])
        return tables

    def Columns(self, table=None):
        c = self.get_connection()
        table = table or self.table
        return c.columns(table)
    
    def ListTables(self):
        l = []
        for t in self.Tables(): l.append(t['TABLE_NAME'])
        return l

    def column_list(self):
        l = []
        for column in self.column_order:
            c = getattr(self, column)
            l.append(c)
        return l
        
    def SELECT_columns(self):
        from string import join
        l = []
        for c in self.column_list():
            l.append(c.read_column_reference())
        return join(l, ', ')

    def SELECT_args(self):
        l = []
        for c in self.column_list():
            if not hasattr(c,'INDEX'):
                raise "WTF", (self.table,c)
            if c.INDEX:
                l.append(c.ZSQL_SELECT_arg())
        return l

    def WHERE_clause(self):
        from string import join
        l = [self.WHERE]
        for c in self.column_list():
            if c.INDEX:
                l.append(c.sqltest())
        conditions = join(filter(None, l), '\n<dtml-and>\n')
        if conditions:
            return "<dtml-sqlgroup where>\n" \
                   "%s\n" \
                   "</dtml-sqlgroup>" % conditions
        else:
            return ''

    def FROM_clause(self):
        return "FROM %s" % (self.FROM or self.table)
    
    def SELECT_stmt(self):
        "Returns a SELECT statement."
        from string import join
        stmt = join(['SELECT',
                     self.SELECT_columns(),
                     self.FROM_clause(),
                     self.WHERE_clause(),
                     ], '\n')
        return stmt

    def INSERT_columns(self, table):
        from string import join
        c = []
        v = []
        for column in self.column_list():
            if not column.table or (column.table == table):
                if not column.read_only:
                    if column.NOT_NULL:
                        c.append(column.column)
                    else:
                        c.append("<dtml-if %s>%s</dtml-if>" % \
                                 (column.column, column.column))
                    v.append(column.INSERT_sqlvar())
        return "<dtml-sqlgroup required>%s</dtml-sqlgroup>\n" \
               "VALUES <dtml-sqlgroup required>\n%s\n</dtml-sqlgroup>" \
               % (join(c,'<dtml-comma>\n'),
                  join(v,'<dtml-comma>\n'))
    
    def INSERT_args(self, table=None):
        l = []
        for c in self.column_list():
            if not c.read_only and (not c.table or (c.table == table)):
                l.append(c.ZSQL_INSERT_arg())
        return l
    
    def INSERT_stmt(self, table=None):
        "Returns an INSERT statement."
        from string import join
        table = table or self.table
        stmt = join(['INSERT INTO %s' % table,
                     self.INSERT_columns(table),
                     ], '\n')
        return stmt

    def primary_key(self):
        for c in self.column_list():
            if c.primary_key: return c
        return None

    def UPDATE_args(self, table=None):
        l = []
        for c in self.column_list():
            if (not c.read_only or c.primary_key) \
               and (not c.table or (c.table == table)):
                l.append(c.ZSQL_UPDATE_arg())
        return l
        
    def UPDATE_columns(self, table):
        from string import join
        l = []
        for column in self.column_list():
            if not column.table or (column.table == table):
                if not column.read_only:
                    l.append(column.sqltest())
        return "<dtml-sqlgroup set noparens>\n" \
               "%s</dtml-sqlgroup>" % join(l, '<dtml-comma>\n')
            
    def UPDATE_stmt(self, table=None):
        "Returns an UPDATE statement."
        from string import join
        table = table or self.table
        pk = self.primary_key()
        if not pk: return ''
        stmt = join(['UPDATE %s SET' % table,
                     self.UPDATE_columns(table),
                     'WHERE %s' % pk.sqltest(1),
                     ], '\n')
        return stmt

    def table_heading_dtml(self):
        from string import join
        l = ['<TR>']
        for c in self.column_list():
            l.append(c.table_heading())
        l.append('</TR>')
        return join(l,'\n')

    def table_row_dtml(self, even=0):
        from string import join
        start_tag = "<TR>"
        if self.TR_even_odd:
            if even and getattr(self, 'TR_attr_even', None):
                start_tag = "<TR %s>" % self.TR_attr_even
            elif not even and getattr(self, 'TR_attr_odd', None):
                start_tag = "<TR %s>" % self.TR_attr_odd
        elif getattr(self, 'TR_attr', None):
            start_tag = "<TR %s>" % self.TR_attr
        l = [start_tag]
        for c in self.column_list():
            l.append(c.table_value())
        l.append('</TR>')
        return join(l,'\n')

    def list_dtml(self):
        from string import join
        dtml = ["<dtml-in SELECT mapping>",
               "<dtml-if sequence-start>",
               "<table>",
               "<dtml-var table_heading>" ,
               "</dtml-if>",
                ]
        if self.TR_even_odd:
            dtml.extend([
                "<dtml-if sequence-odd><dtml-var table_row_odd></dtml-if>",
                "<dtml-if sequence-even><dtml-var table_row_even></dtml-if>",
                ])
        else:
            dtml.append("<dtml-var table_row>")
        dtml.extend([
            "<dtml-if sequence-end>",
            "</table>",
            "</dtml-if>",
            "</dtml-in>",
            ])
        return join(dtml, '\n')

    def form_dtml(self):
        from string import join
        dtml = [ '<FORM ACTION="edit" METHOD="POST" name="%s_edit">' \
                 % self.table,
                 '<TABLE>',
                 ]
        for c in self.column_list():
            dtml.extend(["<TR>",
                         "<TH>%s</TH>" % c.table_heading_text(),
                         "<TD>%s</TD></TR>" % c.form_input_element(),
                         ])
        dtml.extend(['</TABLE>', '</FORM>'])
        return join(dtml, '\n')

    def show_dtml(self):
        from string import join
        dtml = [ '<TABLE>' ]
        for c in self.column_list():
            dtml.extend(['<TR>',
                         '<TH>%s</TH>' % c.table_heading_text(),
                         '<TD>%s</TD>' % c.dtml_var(),
                         ])
        dtml.extend(['</TABLE>', '</FORM>'])
        return join(dtml, '\n')

    def select_options_query(self, column):
        c1, c2, c3 = column.select_options_columns()
        if not (c1 and c2):
            return '', ''
        query = "SELECT %s, %s FROM %s\n" \
                "<dtml-sqlgroup where>\n" \
                "<dtml-sqltest %s op=like type=string optional>\n" \
                "</dtml-sqlgroup>\n" % \
                (c1, c2, self.table, c2)
        if c3: query = query + "ORDER BY %s" % c3
        args = '%s:string="%%"' % c2
        return args, query

    def select_dtml(self, column, display):
        from string import join
        col = column
        col2 = display
        dtml = ['<dtml-let selected_%s=%s>' \
                % (col, col),
                '<SELECT name="%s">' % col,
                '<OPTION disabled>Select...</OPTION>',
                '<dtml-in SELECT_%s_options mapping>' % col,
                '<OPTION value="&dtml-%s;"' \
                '<dtml-if expr="selected_%s==%s"> selected</dtml-if>>' \
                '&dtml-%s;</OPTION>' \
                % (col,col,col,col2),
                '</dtml-in></SELECT></dtml-let>',
                ]
        return join(dtml,'\n')
                
    
InitializeClass(SQLBlender)
