/*
 * Copyright (C) 2001 Michele Comitini <mcm@initd.net>
 * Copyright (C) 2001 Federico Di Gregorio <fog@debian.org>
 *
 * This file is part of the psycopg module.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * connection.c -- defines the connection object DBAPI 2.0
 * $Id: connection.c,v 1.47.2.2 2002/04/03 08:02:52 fog Exp $
 */

#include <assert.h>
#include "module.h"


/**** UTILITY FUNCTIONS ****/

/* curs_closeall() - close all the cursors */
static void
curs_closeall(connobject *self)
{
    int len, i;
    PyObject *tmpobj;
    PyObject *cursors = self->cursors;

    pthread_mutex_lock(&(self->lock));

    /* close all the children cursors */
    len = PyList_Size(cursors);
    for (i = 0; i < len; i++) {
        tmpobj = PyList_GetItem(cursors, i);
        assert(tmpobj);
        ((cursobject *)tmpobj)->closed = 1;
    }
    pthread_mutex_unlock(&(self->lock));
}


/* curs_commitall() - execute a commit on all the cursors */
static void
curs_commitall(connobject *self)
{
    int len, i;
    PyObject *cursors = self->cursors;
    cursobject *cursor;
    
    Dprintf("curs_commitall(): acquiring lock\n");
    pthread_mutex_lock(&(self->lock));
    Dprintf("curs_commitall(): lock acquired\n");
    Py_BEGIN_ALLOW_THREADS;
    
    len = PyList_Size(cursors);
    Dprintf("curs_commitall(): commit on %d cursors\n", len);
    
    /* acquire all the required locks */
    for (i = 0; i < len; i++) {
        cursor = (cursobject *)PyList_GetItem(cursors, i);
        assert(cursor);
        if (cursor->keeper->status == KEEPER_BEGIN && cursor->autocommit == 0){
            pthread_mutex_lock(&(cursor->keeper->lock));
            if (cursor->keeper->status == KEEPER_BEGIN) {
                cursor->keeper->status = KEEPER_CONN_LOCK;
                Dprintf("curs_commitall(): acquired lock on keeper %p\n",
                        cursor->keeper);
            }
            else {
                pthread_mutex_unlock(&(cursor->keeper->lock));
            }
        }
    }

    /* does all the commits */
    for (i = 0; i < len; i++) {
        cursor = (cursobject *)PyList_GetItem(cursors, i);
        assert(cursor);
        if (cursor->keeper->status == KEEPER_CONN_LOCK) {
            cursor->keeper->status = KEEPER_BEGIN;
            commit_pgconn(cursor);
            cursor->keeper->status = KEEPER_CONN_READY;
        }
    }

    /* unlocks all the connections */
    for (i = 0; i < len; i++) {
        cursor = (cursobject *)PyList_GetItem(cursors, i);
        assert(cursor);
        if (cursor->keeper->status == KEEPER_CONN_READY) {
            pthread_mutex_unlock(&(cursor->keeper->lock));
            cursor->keeper->status = KEEPER_READY;
            Dprintf("curs_commitall(): released lock on keeper %p\n",
                    cursor->keeper);
        }
    }

    pthread_mutex_unlock(&(self->lock));
    Dprintf("curs_commitall(): lock released\n");
    Py_END_ALLOW_THREADS;
}


/* curs_rollbackall() execute a rollback on all the cursors */
static void
curs_rollbackall(connobject *self)
{
    int len, i;
    PyObject *cursors = self->cursors;
    cursobject *cursor;
    
    Dprintf("curs_rollbackall(): acquiring lock\n");
    pthread_mutex_lock(&(self->lock));
    Dprintf("curs_rollbackall(): lock acquired\n");
    Py_BEGIN_ALLOW_THREADS;
    
    len = PyList_Size(cursors);

    /* acquire all the required locks */
    for (i = 0; i < len; i++) {
        cursor = (cursobject *)PyList_GetItem(cursors, i);
        assert(cursor);
        if (cursor->keeper->status == KEEPER_BEGIN && cursor->autocommit == 0){
            pthread_mutex_lock(&(cursor->keeper->lock));
            if (cursor->keeper->status == KEEPER_BEGIN) {
                cursor->keeper->status = KEEPER_CONN_LOCK;
                Dprintf("curs_commitall(): acquired lock on keeper %p\n",
                        cursor->keeper);
            }
            else {
                pthread_mutex_unlock(&(cursor->keeper->lock));
            }
         }
    }

    /* does all the rollbacks */
    for (i = 0; i < len; i++) {
        cursor = (cursobject *)PyList_GetItem(cursors, i);
        assert(cursor);
        if (cursor->keeper->status == KEEPER_CONN_LOCK) {
            cursor->keeper->status = KEEPER_BEGIN;
            abort_pgconn(cursor);
            cursor->keeper->status = KEEPER_CONN_BEGIN;
        }
    }

    /* unlocks all the connections */
    for (i = 0; i < len; i++) {
        cursor = (cursobject *)PyList_GetItem(cursors, i);
        assert(cursor);
        if (cursor->keeper->status == KEEPER_CONN_BEGIN) {
            pthread_mutex_unlock(&(cursor->keeper->lock));
            cursor->keeper->status = KEEPER_READY;
        }
    }

    pthread_mutex_unlock(&(self->lock));
    Dprintf("curs_rollbackall(): lock released\n");
    Py_END_ALLOW_THREADS;
}


/**** CONNECTION METHODS *****/

/* psyco_conn_close() - close the connection */

static char psyco_conn_close__doc__[] = "Closes the connection.";

static void
_psyco_conn_close(connobject *self)
{
    int len, i;
    PyObject *tmpobj = NULL;
    connkeeper *keeper;
    
    Dprintf("_psyco_conn_close(): shutting down postgres connections\n");

    curs_closeall(self);

    /* orphans all the children cursors but do NOT destroy them (note that
       we need to lock the keepers used by the cursors before destroying the
       connection, else we risk to be inside an execute while we set pgconn
       to NULL) */
    assert(self->cursors != NULL);
    len = PyList_Size(self->cursors);
    Dprintf("_psyco_conn_close(): len(self->cursors) = %d\n", len);
    for (i = 0; i < len; i++) {
        tmpobj = PyList_GetItem(self->cursors, 0);
        assert(tmpobj);
        Py_INCREF(tmpobj);
        PySequence_DelItem(self->cursors, 0);
        ((cursobject *)tmpobj)->conn = NULL; /* orphaned */
        Dprintf("_psyco_conn_close(): cursor refcnt = %d\n",
                tmpobj->ob_refcnt);
    }

    /* close all the open postgresql connections */
    len = PyList_Size(self->avail_conn);
    Dprintf("_psyco_conn_close(): len(self->avail_conn) = %d\n", len);
    for (i = 0; i < len; i++) {
        tmpobj = PyList_GetItem(self->avail_conn, 0);
        assert(tmpobj);
        Py_INCREF(tmpobj);
        if ((keeper = (connkeeper *)PyCObject_AsVoidPtr(tmpobj))) {
            Dprintf("_psyco_conn_close(): destroying avail_conn[%i] at %p\n",
                    i, keeper);
            if(keeper) {
                PQfinish(keeper->pgconn);
                pthread_mutex_destroy(&(keeper->lock));
                free(keeper);
            }
        }
        PySequence_DelItem(self->avail_conn, 0);
        Py_DECREF(tmpobj);
    }
    
    Py_DECREF(self->cursors);
    Py_DECREF(self->avail_conn);

    /* orphan default cursor and destroy it (closing the last connection to the
       database) */
    self->stdmanager->conn = NULL;
    Py_DECREF(self->stdmanager);
}

static PyObject *
psyco_conn_close(connobject *self, PyObject *args)
{
    EXC_IFCLOSED(self);
    PARSEARGS(args);

    /* from now on the connection is considered closed */
    self->closed = 1;  
    _psyco_conn_close(self);
    Dprintf("psyco_conn_close(): connection closed\n");
    Py_INCREF(Py_None);
    return Py_None;
}


/* psyco_conn_commit() - commit the connection (commit all the cursors) */

static char psyco_conn_commit__doc__[] =
"Commit any pending transaction to the database.";

static PyObject *
psyco_conn_commit(connobject *self,PyObject *args)
{
    EXC_IFCLOSED(self);
    PARSEARGS(args);
    curs_commitall(self);
    Dprintf("psyco_conn_commit(): commit executed\n");
    Py_INCREF(Py_None);
    return Py_None;
}


/* psyco_conn_rollback() - rollback the connection (rollback on all cursors) */

static char psyco_conn_rollback__doc__[] = 
"Causes the database to roll back to the start of any pending transaction.";

static PyObject *
psyco_conn_rollback(connobject *self,	PyObject *args)
{
    EXC_IFCLOSED(self);
    PARSEARGS(args);
    curs_rollbackall(self);
    Dprintf("psyco_conn_rollback(): rollback executed\n");
    Py_INCREF(Py_None);
    return Py_None;
}


/* psyco_conn_cursor() - create a new cursor */

static char psyco_conn_cursor__doc__[] = 
"Return a new Cursor Object using the connection.";

static PyObject *
psyco_conn_cursor(connobject *self, PyObject *args)
{
    char *name = NULL;
    connkeeper *keeper = NULL;
    PyObject *obj;
    
    if (!PyArg_ParseTuple(args, "|s", &name)) {
        return NULL;
    }
    
    EXC_IFCLOSED(self);

    Dprintf("psyco_conn_cursor(): conn = %p, name = %s\n", self, name);
    Dprintf("psyco_conn_cursor(): serialize = %d, keeper = %p\n",
            self->serialize, self->stdmanager->keeper);
    
    if (self->serialize == 0 || name) {
        keeper = NULL;
    }
    else {
        /* here we need to bump up the refcount before passing the keeper
           to the cursor for initialization, because another thread can
           dispose the keeper in the meantime */
        keeper = self->stdmanager->keeper;
        pthread_mutex_lock(&(keeper->lock));
        keeper->refcnt++;
        pthread_mutex_unlock(&(keeper->lock));
    }
    
    obj = (PyObject *)new_psyco_cursobject(self, keeper);
    return obj; 
}


/* psyco_conn_autocommit() - put connection (and all derived cursors)
   in autocommit mode */

static char psyco_conn_autocommit__doc__[] = 
"Switch connection (and derived cursors) to/from autocommit mode.";

static PyObject *
psyco_conn_autocommit(connobject *self, PyObject *args)
{
    PyObject *tmpobj;
    long int ac = 1; /* the default is to set autocommit on */
    int i, len;
    
    if (!PyArg_ParseTuple(args, "|l", &ac)) {
        return NULL;
    }

    /* save the value */
    self->autocommit = (int)ac;

    /* call autocommit(ac) on all the cursors to change their status too */
    len = PyList_Size(self->cursors);
    for (i = 0; i < len; i++) {
        tmpobj = PyList_GetItem(self->cursors, i);
        assert(tmpobj);
        curs_switch_autocommit((cursobject *)tmpobj, ac);
    }

    Py_INCREF(Py_None);
    return Py_None;
}

/* psyco_conn_serialize() - switch on and off cursor serialization */

static char psyco_conn_serialize__doc__[] = 
"Switch on and off cursor serialization.";

static PyObject *
psyco_conn_serialize(connobject *self, PyObject *args)
{
    long int se = 1; /* the default is to set serialize on */
    
    if (!PyArg_ParseTuple(args, "|l", &se)) {
        return NULL;
    }

    /* save the value */
    self->serialize = (int)se;

    Py_INCREF(Py_None);
    return Py_None;
}


/**** CONNECTION OBJECT DEFINITION ****/

/* object methods list */

static struct PyMethodDef psyco_conn_methods[] = {
    {"close", (PyCFunction)psyco_conn_close,
     METH_VARARGS, psyco_conn_close__doc__},
    {"commit", (PyCFunction)psyco_conn_commit,
     METH_VARARGS, psyco_conn_commit__doc__},
    {"rollback", (PyCFunction)psyco_conn_rollback,
     METH_VARARGS, psyco_conn_rollback__doc__},
    {"cursor", (PyCFunction)psyco_conn_cursor,
     METH_VARARGS, psyco_conn_cursor__doc__},
    {"autocommit", (PyCFunction)psyco_conn_autocommit,
     METH_VARARGS, psyco_conn_autocommit__doc__},
    {"serialize", (PyCFunction)psyco_conn_serialize,
     METH_VARARGS, psyco_conn_serialize__doc__},
    {NULL, NULL}
};


/* object member list */

#define OFFSETOF(x) offsetof(connobject, x)

static struct memberlist psyco_conn_memberlist[] = {
    { "cursors", T_OBJECT, OFFSETOF(cursors), RO},
    { "maxconn", T_INT, OFFSETOF(maxconn), RO},
    { "minconn", T_INT, OFFSETOF(minconn), RO},
    {NULL}
};


/* the python object interface for the connection object */

static PyObject *
psyco_conn_getattr(connobject *self, char *name)
{
    PyObject *rv;
	
    rv = PyMember_Get((char *)self, psyco_conn_memberlist, name);
    if (rv) return rv;
    PyErr_Clear();
    return Py_FindMethod(psyco_conn_methods, (PyObject *)self, name);
}

static int
psyco_conn_setattr(connobject *self, char *name, PyObject *v)
{
    if (v == NULL) {
        PyErr_SetString(PyExc_AttributeError, "cannot delete attribute");
        return -1;
    }
    return PyMember_Set((char *)self, psyco_conn_memberlist, name, v);
}

static void
psyco_conn_destroy(connobject *self)
{
    _psyco_conn_close(self);
    pthread_mutex_destroy(&(self->lock));
    free(self->dsn);
    PyObject_Del(self);
    Dprintf("psyco_conn_destroy(): connobject at %p destroyed\n", self);
}


static char Conntype__doc__[] = "Connections Object.";

static PyTypeObject Conntype = {
    PyObject_HEAD_INIT(&PyType_Type)
    0,				        /*ob_size*/
    "connection",			/*tp_name*/
    sizeof(connobject),		/*tp_basicsize*/
    0,				        /*tp_itemsize*/

    /* methods */
    (destructor)psyco_conn_destroy,   /*tp_dealloc*/
    (printfunc)0,		             /*tp_print*/
    (getattrfunc)psyco_conn_getattr,	 /*tp_getattr*/
    (setattrfunc)psyco_conn_setattr,	 /*tp_setattr*/
    (cmpfunc)0,		                 /*tp_compare*/
    (reprfunc)0,		             /*tp_repr*/
    0,			                     /*tp_as_number*/
    0,                               /*tp_as_sequence*/
    0,		                         /*tp_as_mapping*/
    (hashfunc)0,                     /*tp_hash*/
    (ternaryfunc)0,		             /*tp_call*/
    (reprfunc)0,                     /*tp_str*/
  
    /* Space for future expansion */
    0L,0L,0L,0L,
    Conntype__doc__ /* Documentation string */
};


/* the C constructor for connection objects */

connobject *
new_psyco_connobject(char *dsn, int maxconn, int minconn, int serialize)
{
    connobject *self;

    Dprintf("new_psyco_connobject(): creating new connection\n");
  
    self = PyObject_NEW(connobject, &Conntype);
    if (self == NULL) return NULL;

    pthread_mutex_init(&(self->lock), NULL);
    self->dsn = strdup(dsn);
    self->maxconn = maxconn;       
    self->minconn = minconn;
    self->cursors = PyList_New(0);
    self->avail_conn = PyList_New(0);
    self->closed = 0;
    self->autocommit = 0;
    self->serialize = serialize;
    
    /* allocate default manager thread and keeper */
    self->stdmanager = new_psyco_cursobject(self, NULL);

    /* error checking done good */
    if (self->stdmanager == NULL || self->cursors == NULL
        || self->avail_conn == NULL) {
        Py_XDECREF(self->cursors);
        Py_XDECREF(self->avail_conn);
        Py_XDECREF(self->stdmanager);
        pthread_mutex_destroy(&(self->lock));
        PyObject_Del(self);
        return NULL;
    }

    Dprintf("new_psyco_connobject(): created connobject at %p, refcnt = %d\n",
            self, self->ob_refcnt);
    Dprintf("new_psyco_connobject(): stdmanager = %p, stdkeeper = %p\n",
            self->stdmanager, self->stdmanager->keeper);
    return self;
}




