##############################################################################
#
# Zope Public License (ZPL) Version 1.0
# -------------------------------------
#
# Copyright (c) Digital Creations.  All rights reserved.
#
# This license has been certified as Open Source(tm).
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
# 1. Redistributions in source code must retain the above copyright
#    notice, this list of conditions, and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions, and the following disclaimer in
#    the documentation and/or other materials provided with the
#    distribution.
#
# 3. Digital Creations requests that attribution be given to Zope
#    in any manner possible. Zope includes a "Powered by Zope"
#    button that is installed by default. While it is not a license
#    violation to remove this button, it is requested that the
#    attribution remain. A significant investment has been put
#    into Zope, and this effort will continue if the Zope community
#    continues to grow. This is one way to assure that growth.
#
# 4. All advertising materials and documentation mentioning
#    features derived from or use of this software must display
#    the following acknowledgement:
#
#      "This product includes software developed by Digital Creations
#      for use in the Z Object Publishing Environment
#      (http://www.zope.org/)."
#
#    In the event that the product being advertised includes an
#    intact Zope distribution (with copyright and license included)
#    then this clause is waived.
#
# 5. Names associated with Zope or Digital Creations must not be used to
#    endorse or promote products derived from this software without
#    prior written permission from Digital Creations.
#
# 6. Modified redistributions of any form whatsoever must retain
#    the following acknowledgment:
#
#      "This product includes software developed by Digital Creations
#      for use in the Z Object Publishing Environment
#      (http://www.zope.org/)."
#
#    Intact (re-)distributions of any official Zope release do not
#    require an external acknowledgement.
#
# 7. Modifications are encouraged but must be packaged separately as
#    patches to official Zope releases.  Distributions that do not
#    clearly separate the patches from the original work must be clearly
#    labeled as unofficial distributions.  Modifications which do not
#    carry the name Zope may be packaged in any form, as long as they
#    conform to all of the clauses above.
#
#
# Disclaimer
#
#   THIS SOFTWARE IS PROVIDED BY DIGITAL CREATIONS ``AS IS'' AND ANY
#   EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
#   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
#   PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL DIGITAL CREATIONS OR ITS
#   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
#   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
#   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
#   ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
#   OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
#   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
#   SUCH DAMAGE.
#
#
# This software consists of contributions made by Digital Creations and
# many individuals on behalf of Digital Creations.  Specific
# attributions are listed in the accompanying credits file.
#
##############################################################################
__doc__='''Define Zope\'s default security policy


$Id: DebugSecurityPolicy.py,v 1.4 2000/09/07 02:34:01 shane Exp $'''
__version__='$Revision: 1.4 $'[11:-2]

from AccessControl import SimpleObjectPolicies
_noroles=[]

from AccessControl.PermissionRole import \
     _what_not_even_god_should_do, rolesForPermissionOn
from string import join


class DebugSecurityPolicy:

    def validate(self, accessed, container, name, value, context,
                 None=None, type=type, IntType=type(0), DictType=type({}),
                 getattr=getattr, _noroles=_noroles, StringType=type(''),
                 Containers=SimpleObjectPolicies.Containers,
                 valid_aq_=('aq_parent','aq_explicit')):


        ############################################################
        # Provide special rules for the acquisition attributes
        if type(name) is StringType:
            if name[:3]=='aq_' and name not in valid_aq_:
                return 0

        containerbase=getattr(container, 'aq_base', container)
        accessedbase=getattr(accessed, 'aq_base', container)

        ############################################################
        # Try to get roles
        roles=getattr(value, '__roles__', _noroles)

        if roles is _noroles:

            ############################################################
            # We have an object without roles. Presumably, it's
            # some simple object, like a string or a list.
            if container is None: return 0 # Bail if no container

            roles=getattr(container, '__roles__', _noroles)
            if roles is _noroles:
                aq=getattr(container, 'aq_acquire', None)
                if aq is None:
                    roles=_noroles
                    if containerbase is not accessedbase: return 0
                else:
                    # Try to acquire roles
                    try: roles=aq('__roles__')
                    except AttributeError:
                        roles=_noroles
                        if containerbase is not accessedbase: return 0

            # We need to make sure that we are allowed to
            # get unprotected attributes from the container. We are
            # allowed for certain simple containers and if the
            # container says we can. Simple containers
            # may also impose name restrictions.
            p=Containers(type(container), None)
            if p is None:
                p=getattr(container,
                          '__allow_access_to_unprotected_subobjects__', None)

            if p is not None:
                tp=type(p)
                if tp is not IntType:
                    if tp is DictType:
                        p=p.get(name, None)
                    else:
                        p=p(name, value)

            if not p:
                if (containerbase is accessedbase):
                    info = 'Access denied for %s because ' \
                           'its container, %s, has no security ' \
                           'assertions.' % \
                           (cleanupName(name, value),
                            getCleanPath(container, value))
                    raise 'Unauthorized', info
                else:
                    return 0

            if roles is _noroles: return 1

            # We are going to need a security-aware object to pass
            # to allowed(). We'll use the container.
            value=container

        # Short-circuit tests if we can:
        if roles is None or 'Anonymous' in roles: return 1

        # Check executable security
        stack=context.stack
        if stack:
            eo=stack[-1]

            # If the executable had an owner, can it execute?
            owner=eo.getOwner()
            if (owner is not None) and not owner.allowed(value, roles):
                # We don't want someone to acquire if they can't
                # get an unacquired!
                if accessedbase is containerbase:
                    if len(roles) < 1:
                        info = 'The object "%s", which is contained in ' \
                               '%s, is marked as private.' % (
                            cleanupName(name, value),
                            getCleanPath(container, value))
                    elif userHasRolesButNotInContext(owner, value, roles):
                        info = 'User %s, who is the owner of %s, is ' \
                               'defined in the context of %s, which is not ' \
                               'an ancestor of %s. ' \
                               % (
                            owner.getUserName(), getCleanPath(eo),
                            getCleanPath(owner.aq_parent.aq_parent),
                            cleanupName(name, value),
                            #aqChain(owner.aq_parent.aq_parent),
                            #aqChain(value),
                            )
                    else:
                        info = 'User %s, who is the owner of %s, is not ' \
                               'allowed to access ' \
                               '%s, which is contained in %s. The owner, ' \
                               'whose roles are %s, would need to have ' \
                               'one of the following roles: %s.' % (
                            owner.getUserName(),
                            getCleanPath(eo), cleanupName(name, value),
                            getCleanPath(container, value),
                            tuple(owner.getRolesInContext(value)),
                            tuple(roles))
                    raise 'Unauthorized', info
                return 0

            # Proxy roles, which are a lot safer now.
            proxy_roles=getattr(eo, '_proxy_roles', None)
            if proxy_roles:
                for r in proxy_roles:
                    if r in roles: return 1

                # Proxy roles actually limit access!
                if accessedbase is containerbase:
                    if len(roles) < 1:
                        info = 'The object %s, which is contained in ' \
                               '"%s", is marked as private.' % (
                            cleanupName(name, value),
                            getCleanPath(container, value))
                    else:
                        info = 'The proxy roles set for ' \
                               '%s do not allow access to %s, which is ' \
                               'contained in %s. The proxy roles would ' \
                               'have to include one of the following ' \
                               'roles: %s' % (
                            getCleanPath(eo), cleanupName(name, value),
                            getCleanPath(container, value),
                            tuple(roles))
                    raise 'Unauthorized', info

                return 0


        try:
            if context.user.allowed(value, roles): return 1
        except AttributeError: pass

        # We don't want someone to acquire if they can't get an unacquired!
        if accessedbase is containerbase:
            if len(roles) < 1:
                info = 'The object %s, ' \
                       'which is contained in %s, is marked as private.' % (
                    cleanupName(name, value),
                    getCleanPath(container, value))
            elif userHasRolesButNotInContext(context.user, value, roles):
                info = 'User %s is ' \
                       'defined in the context of %s, which is not ' \
                       'an ancestor of %s.' \
                       % (
                    context.user.getUserName(),
                    getCleanPath(context.user.aq_parent.aq_parent),
                    cleanupName(name, value),
                    #aqChain(context.user.aq_parent.aq_parent),
                    #aqChain(value),
                    )
            else:
                info = 'User %s is ' \
                       'not allowed to access %s, which is contained ' \
                       'in %s. The user, whose roles are %s, would ' \
                       'need to have one of the following roles: %s.' % (
                    context.user.getUserName(), cleanupName(name, value),
                    getCleanPath(container, value),
                    tuple(context.user.getRolesInContext(value)),
                    tuple(roles))
            raise 'Unauthorized', info

        return 0

    def checkPermission(self, permission, object, context):
        roles=rolesForPermissionOn(permission, object)
        if roles is _what_not_even_god_should_do: return 0
        return context.user.has_role(roles, object)


def cleanupName(name, value):
    # If value is available, does getCleanPath().
    if value is not None:
        return getCleanPath(value)
    return name

def getPathOf(object):
    # Mostly copied from Zope/__init__.py .
    if hasattr(object, 'getPhysicalPath'):
        path = join(object.getPhysicalPath(), '/')
    else:
        # Try hard to get the physical path of the object,
        # but there are many circumstances where that's not possible.
        to_append = ()

        if hasattr(object, 'im_self') and hasattr(object, '__name__'):
            # object is a Python method.
            to_append = (object.__name__,)
            object = object.im_self

        while object is not None and \
              not hasattr(object, 'getPhysicalPath'):
            if not hasattr(object, '__name__'):
                object = None
                break
            to_append = (object.__name__,) + to_append
            object = getattr(object, 'aq_inner', object)
            object = getattr(object, 'aq_parent', None)

        if object is not None:
            path = join(object.getPhysicalPath() + to_append, '/')
        else:
            # Couldn't do it.  Try to just get the id.
            try:
                path = object.id
            except:
                path = str(object)
            else:
                try: path = path()
                except: pass
    return path

def getCleanPath(obj, child=None):
    if obj is None:
        if hasattr(child, 'im_self'):
            obj = getattr(child, 'im_self')
        else:
            obj = getattr(child, 'aq_inner', child)
            obj = getattr(child, 'aq_parent', child)
    name = getPathOf(obj)
    return name

def userHasRolesButNotInContext(user, object, roles):
    '''Returns 1 if the user has any of the listed roles but
    is not defined in a context which is not an ancestor of object.
    '''
    if roles is None or 'Anonymous' in roles:
        return 0
    usr_roles=user.getRolesInContext(object)
    for role in roles:
        if role in usr_roles:
            # User has the roles.
            return (not verifyAcquisitionContext(user, object, roles))
    return 0

def verifyAcquisitionContext(self, parent, roles=None):
    """Mimics the relevant section of User.allowed(). self is a user object.
    """
    inner_user = getattr(self, 'aq_inner', self)
    userfolder = getattr(inner_user, 'aq_parent', None)
    inner_userfolder = getattr(userfolder, 'aq_inner', userfolder)
    usercontext = getattr(inner_userfolder, 'aq_parent', None)
    if usercontext is not None:
        if parent is None: return 1
        if (not hasattr(parent, 'aq_inContextOf') and
            hasattr(parent, 'im_self')):
            # This is a method, grab it's self.
            parent=parent.im_self
        if not parent.aq_inContextOf(usercontext,1):
            if 'Shared' in roles:
                # Darn, old role setting. Waaa
                roles=self._shared_roles(parent)
                if 'Anonymous' in roles: return 1
            return None
    return 1

def aqChain(object):
    # Returns a representation of the acquisition chain of an object.
    obj = object
    rval = []
    while obj is not None:
        base = getattr(obj, 'aq_base', obj)
        rval.append(type(base))
        obj = getattr(obj, 'aq_parent', None)
        obj = getattr(obj, 'aq_inner', obj)
    rval.append(object.aq_chain)
    return rval
