#!/usr/bin/python

# This software is released under the following Wide-Open Source licence:
#
# Copyright (c) 2000 Shane Hathaway
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. Neither the name of the Author nor the names of other contributors
#    may be used to endorse or promote products derived from this software
#    without specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.

# This program depends on two modules originally provided by Medusa
# but which are now part of the main Python distribution, asynchat and
# asyncore.  There are many versions of asynchat and asyncore floating
# around; if you have troubles you might try going to
# http://www.nightmare.com/medusa/index.html

import asynchat
import asyncore
import socket
import string

from Tkinter import *
from thread import start_new_thread
from time import time, localtime
from cStringIO import StringIO
import asyncore
from ScrolledText import ScrolledText

# This program has two distinct parts: the proxy server and the GUI.

class proxy_server (asyncore.dispatcher):
    '''Accepts connections and uses a factory to open streams.
    '''
    
    def __init__ (self, localport, host, port, factory):
        self.factory = factory
        asyncore.dispatcher.__init__ (self)
        self.create_socket (socket.AF_INET, socket.SOCK_STREAM)
        self.set_reuse_addr()
        self.there = (host, port)
        here = ('', localport)
        self.bind (here)
        self.listen (5)

    def handle_accept (self):
        self.factory (self, self.accept())


class proxy_sender (asynchat.async_chat):
    '''Forwards data to the proxy_receiver while allowing the
    receiver object to capture the data.
    '''

    def __init__ (self, receiver, address):
        asynchat.async_chat.__init__ (self)
        self.receiver = receiver
        self.set_terminator (None)
        self.create_socket (socket.AF_INET, socket.SOCK_STREAM)
        self.buffer = ''
        self.connect (address)

    def handle_connect (self):
        self.receiver.watch_connect(0)

    def collect_incoming_data (self, data):
        if self.receiver is not None:
            self.receiver.push (data)
        self.buffer = self.receiver.linesplit_watch(self.buffer + data, 0)

    def cleanupRefs(self):
        self.receiver = None

    def handle_close (self):
        if len(self.buffer) > 0:
            self.receiver.watch_output(self.buffer, 0)
            self.buffer = ''
        self.receiver.watch_close(0)
        self.receiver.close_when_done()
        self.close()
        # Remove cyclic references.
        self.receiver.cleanupRefs()
        self.cleanupRefs()

class proxy_receiver (asynchat.async_chat):
    '''Sends data to the sender while capturing the data for monitoring.
    '''

    def __init__ (self, server, (conn, addr)):
        asynchat.async_chat.__init__ (self, conn)
        self.set_terminator (None)
        self.server = server
        self.sender = proxy_sender (self, server.there)
        self.buffer = ''
        self.watch_connect(1)

    def collect_incoming_data (self, data):
        if self.sender is not None:
            self.sender.push(data)
        self.buffer = self.linesplit_watch(self.buffer + data, 1)

    def linesplit_watch(self, data, byClient):
        pos = 0
        while 1:
            oldpos = pos
            pos = string.find(data, '\n', oldpos)
            if pos >= 0:
                line = data[oldpos:pos + 1]
                self.watch_output(line, byClient)
                pos = pos + 1
            else:
                # Last line, may be incomplete.
                return data[oldpos:]
        
    def cleanupRefs(self):
        self.sender = None
        self.server = None

    def handle_close (self):
        if len(self.buffer) > 0:
            self.watch_output(self.buffer, 1)
            self.buffer = ''
        self.watch_close(1)
        self.sender.close_when_done()
        self.close()
        # Remove cyclic references.
        self.sender.cleanupRefs()
        self.cleanupRefs()

    def watch_output(self, data, byClient):
        arrow = (byClient and '==>' or '<==')
        print ('%s %s' % (arrow, data)),

    def watch_connect(self, byClient):
        print '%s connected' % (byClient and 'Client' or 'Server')

    def watch_close(self, byClient):
        print 'Closed by ' + (byClient and 'client' or 'server')

# GUI begins here.

class TCPWatch (Frame):
    '''The tcpwatch top-level window.
    '''
    def __init__(self, master=None):
        Frame.__init__(self, master)
        self.createWidgets()
        self.pack()
        self.connections = {}  # Maps ids to ProxyConnections or strings.
        self.counter = 0
        self.showingid = ''

    def createWidgets(self):
        listframe = Frame(self)
        listframe.pack(side=LEFT, fill=BOTH, expand=1)
        scrollbar = Scrollbar(listframe, orient=VERTICAL)
        self.connectlist = Listbox(listframe, yscrollcommand=scrollbar.set,
                                   exportselection=0)
        scrollbar.config(command=self.connectlist.yview)
        scrollbar.pack(side=RIGHT, fill=Y)
        self.connectlist.pack(side=LEFT, fill=BOTH, expand=1)
        self.connectlist.bind('<Button-1>', self.mouseListSelect)
        self.textbox = ScrolledText(self)
        self.textbox.pack(side='right', fill=BOTH, expand=1)

    def addConnection(self, id, conn):
        self.connections[id] = conn
        connectlist = self.connectlist
        connectlist.insert('end', id)

    def updateConnection(self, id, info, finalUpdate=0):
        if id == self.showingid:
            textbox = self.textbox
            textbox.insert('end', info)
            if finalUpdate:
                # Convert the connection info to an immutable string.
                self.connections[id] = self.connections[id].getBufferValue()

    def mouseListSelect(self, event = None):
        connectlist = self.connectlist
        idx = connectlist.nearest(event.y)
        sel = connectlist.get(idx)
        connections = self.connections
        if connections.has_key(sel):
            conn = connections[sel]
            if type(conn) != type(''):
                value = conn.getBufferValue()
            else:
                value = conn
            self.showingid = ''
            self.textbox.delete('0.0', 'end')
            self.textbox.insert('end', value)
            self.showingid = sel


class ProxyConnection (proxy_receiver):
    '''A proxy_receiver which forwards captured data to a TCPWatch
    frame.  The data is mangled for presentation.
    '''
    
    def __init__(self, frame, server, c):
        self.__buf = StringIO()
        self.__frame = frame
        self.__start = start = time()
        t = localtime(start)
        frame.counter = counter = frame.counter + 1
        id = '%03d (%02d:%02d:%02d)' % (counter, t[3], t[4], t[5])
        self.__id = id
        proxy_receiver.__init__(self, server, c)
        frame.addConnection(id, self)
    
    def watch_connect(self, byClient):
        if not byClient:
            offset = time() - self.__start
            info = '[%0.03f - Server connected]\n' % offset
            self.__buf.write(info)
            self.__frame.updateConnection(self.__id, info)

    def watch_output(self, data, byClient):
        arrow = (byClient and '==>' or '<==')
        encode = 0
        for c in data:
            if (c < ' ' or c > '~') and c not in ('\t', '\r', '\n'):
                encode = 1
                break
        if encode:
            # Make the data printable.
            encoded = StringIO()
            for c in data:
                if (c < ' ' or c > '~') and c not in ('\t', '\r', '\n'):
                    # Encode this character in decimal.
                    encoded.write('\\%03d' % ord(c))
                elif c == '\\':
                    # Double the backslash.
                    encoded.write('\\\\')
                else:
                    encoded.write(c)
            data = encoded.getvalue()
        info = '%s %s' % (arrow, data)
        self.__buf.write(info)
        self.__frame.updateConnection(self.__id, info)

    def watch_close(self, byClient):
        offset = int((time() - self.__start) * 1000) / 1000.0
        info = '[%f - Closed by %s]\n' % \
               (offset, (byClient and 'Client' or 'Server'))
        self.__buf.write(info)
        self.__frame.updateConnection(self.__id, info, 1)

    def getBufferValue(self):
        return self.__buf.getvalue()
    

if __name__ == '__main__':
    import sys
    import string
    argv = sys.argv
    if len(argv) < 4:
        print 'TCPWatch 0.9 (c) 2000 by Shane Hathaway, Digital Creations'
        print 'http://www.digicool.com'
        print 'Monitors TCP connections through a proxy connection. ' \
              'Requires Tkinter.'
        print 'Usage: %s <local-port> <server-host> <server-port>' % argv[0]
    else:

        def window_loop(app):
            app.mainloop()
            asyncore.close_all()

        localport = string.atoi(argv[1])
        host = argv[2]
        port = string.atoi(argv[3])
        app = TCPWatch()
        try:
            app.master.wm_title(
                'tcpwatch: %d -> %s:%d' % (localport, host, port))
        except:
            pass  # No wm_title available.
        app.pack(fill=BOTH, expand=1)
        def connectionFactory(server, c, app=app):
            return ProxyConnection(app, server, c)
        ps = proxy_server(localport, host, port, connectionFactory)
        window_thread = start_new_thread(window_loop, (app,))
        asyncore.loop(timeout=1.0)
