"""
 (C) Copyright 2002 Kapil Thangavelu <kvthan@wm.edu>
 All Rights Reserved

 This file is part of Gideon.

 Gideon is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 Gideon is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with Gideon; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""

# $Id: $

"""
this class tries to mimic some
transactionality to filesystem
operations.

it depends on two atomic file
operations, rename and unlink

this definitely needs some unit
tests to root out any platform
specific behavior irregularities.

"""

from os import path

import os
import string
from tempfile import mktemp

from Shared.DC.ZRDB.TM import TM

def recovery_protocol(filesystem):

    files = os.listdir(filesystem.tmpdir)

    remove_files = filter(lambda x: x.startswith('rm'), files)
    add_files = filter(lambda x: not x.startswith('rm'), files)

    # remove added files
    map(os.unlink, map(path.join,[filesystem.tmpdir]*len(add_files),
                       add_files))

    # restore deleted files
    map(
        lambda x, s=filesystem.storedir, t=filesystem.tmpdir: \
        os.rename(path.join(t,x), path.join(s,x[2:])),
        remove_files    
        )
        
def check_directories(*dirs):
    
    for d in dirs:
        if not path.exists(d):
            raise TxnDirectoryError("directory path not valid %s"%d)
        if not  path.isdir(d):         
            raise TxnDirectoryError("not valid directory %s"%d)
        if not os.access(d, os.W_OK):
            raise TxnDirectoryError("can't write to  directory %s"%d)
        if not os.access(d, os.R_OK):
            raise TxnDirectoryError("can't read  directory %s"%d)        



class TransactionalFileSystem:

    def __init__(self, storedir, tmpdir, bufsize=2048, recover=1, tracing=0):
        
        check_directories(storedir, tmpdir)

        self.storedir = storedir
        self.tmpdir   = tmpdir
        self.bufsize  = bufsize

        if recover:
            recovery_protocol(self)

        # it was either dup the info or make a cycle.
        self._txncontrol = TransactionalFileSystemRegistration(storedir,
                                                               tmpdir,
                                                               tracing)

    def add_file_object(self, fd, name=''):

        if not name:
            name = getattr(fd, 'name', '')

            if not name:
                name =mktemp()

        name = name.split(os.sep)[-1]
        
        try:
            fd.seek(0,2)
            size = fd.tell()
            fd.seek(0)
        except:
            raise TxnFsFileError("invalid file object")

        fh = open(path.join(self.tmpdir, name), 'w')

        while size:
            buf = fd.read(self.bufsize)
            fh.write(buf)
            size -= len(buf)

        self._txncontrol.add_file(name)

    def add_file(self, filepath):

        if not path.exists(filepath):
            raise TxnFsFileError("bad path to file %s"%filepath)

        fh = open(filepath)

        self.add_file_object(fh)

    def remove_file(self, name):
        
        if not path.exists(path.join(self.storedir, name)):
            raise TxnFsFileError("no such file %s"%name)

        os.rename(path.join(self.storedir, name),
                  path.join(self.tmpdir, 'rm'+name)
                  )

        self._txncontrol.remove_file(name)



class TransactionalFileSystemRegistration(TM):

    def __init__(self, storedir, tempdir, tracing ):
        
        self._added = []
        self._removed = []
        self._registered = 0
        self._final = 0
        self.storedir = storedir
        self.tempdir  = tempdir
        self.tracing = tracing

        
    def add_file(self, name):
        """ register an added file """
        
        self._added.append(name)
        self._register()
        
    def remove_file(self, name):
        """ register a removed file """
        
        self._removed.append(name)
        self._register()

    def finish_add(self):
        """ finish move from tmp to store for added files """
        
        for f in self._added:
            os.rename(path.join(self.tempdir, f),
                      path.join(self.storedir, f))

    def abort_add(self):

        for f in self._added:
            os.unlink(path.join(self.tempdir, f))

    def finish_remove(self):

        for f in self._removed:
            os.unlink(path.join(self.tempdir, 'rm'+f))

    def abort_remove(self):
        
        for f in self._removed:
            os.rename(path.join(self.tempdir, 'rm'+f),
                      path.join(self.storedir, f))

    # probably need to think about this one more...
    """
    def commit_sub(self, *args): pass
    def abort_sub(self, *args): pass
    """

    def tpc_vote(self, *args):
        self._final = 1

    def _abort(self):
        if self.tracing: print 'aborting'
        self.abort_add()
        self.abort_remove()

    def _finish(self):
        if self.tracing: print 'finishing'
        #  don't commit unless we're definitely done
        #  the need for this looks like a bug in TM.py
        if self._final: 
            self.finish_add()
            self.finish_remove()
                           

class TxnFsDirectoryError(Exception): pass

class TxnFsFileError(Exception): pass


