"""
 (C) Copyright 2002 Kapil Thangavelu <kvthan@wm.edu>
 All Rights Reserved

 This file is part of Gideon.

 Gideon is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 Gideon is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with Gideon; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""

# $Id: $

import unittest
import os
import shutil
from os.path import join

from TransactionalFileSystem import TransactionalFileSystem

from ZODB.Transaction import free_transaction

class TransactionalFileSystemTests(unittest.TestCase):
    
    def setUp(self):
        
        shutil.copy(os.sep.join( ('data','gdn.wpr') ),
                    os.sep.join( ('store','gdn.wpr') )
                    )

    def tearDown(self):
        
        map(os.unlink, [join('store',f) for f in os.listdir('store')])
        map(os.unlink, [join('temp',f) for f in os.listdir('temp')])

        # just to clear the transaction
        get_transaction()._init()
        free_transaction()
        
    # redundant
    def testFileAddMoveToTemp(self):

        fs = TransactionalFileSystem('store', 'temp')
        fs.add_file('data/LICENSE.txt')

        assert os.listdir('temp') == ['LICENSE.txt']

    def testFileAddAbort(self):

        fs = TransactionalFileSystem('store', 'temp')

        fs.add_file('data/LICENSE.txt')
        
        get_transaction().abort()
        
        assert os.listdir('store') == ['gdn.wpr']
        assert os.listdir('temp') == []

    def testFileAddCommit(self):

        fs = TransactionalFileSystem('store', 'temp')
        
        fs.add_file('data/LICENSE.txt')

        get_transaction().commit()
        
        assert os.listdir('store') == ['gdn.wpr', 'LICENSE.txt']

    def testFileAddCommitSub(self):

        fs = TransactionalFileSystem('store', 'temp', tracing=1)
        fs.add_file('data/LICENSE.txt')

        get_transaction().commit(1)

        assert os.listdir('store') == ['gdn.wpr']
        assert os.listdir('temp') == ['LICENSE.txt']

    def testFileAddAbortSub(self):

        fs = TransactionalFileSystem('store', 'temp')
        fs.add_file('data/LICENSE.txt')

        get_transaction().commit(1)
        get_transaction().abort()

        assert os.listdir('store') == ['gdn.wpr']
        assert os.listdir('temp') == []        

    def testFileObjAddWithNameCommit(self):

        fs = TransactionalFileSystem('store', 'temp')

        import cStringIO
        fh = cStringIO.StringIO(open('data/LICENSE.txt').read())
        
        fs.add_file_object(fh, 'baz')

        get_transaction().commit()

        assert os.listdir('store') == ['gdn.wpr', 'baz']
        assert os.listdir('temp') == []        

    # redundant
    def testFileRemoveMoveToTemp(self):

        fs = TransactionalFileSystem('store', 'temp')

        fs.remove_file('gdn.wpr')

        assert os.listdir('store') == []
        assert os.listdir('temp') == ['rmgdn.wpr']        

    def testFileRemoveCommit(self):

        fs = TransactionalFileSystem('store', 'temp')

        fs.remove_file('gdn.wpr')

        get_transaction().commit()

        assert os.listdir('store') == []
        assert os.listdir('temp') == []

    def testFileRemoveAbort(self):

        fs = TransactionalFileSystem('store', 'temp')

        fs.remove_file('gdn.wpr')

        get_transaction().abort()

        assert os.listdir('store') == ['gdn.wpr']        
        assert os.listdir('temp') == []

    def testRecoveryProtcol(self):

        fs = TransactionalFileSystem('store', 'temp')
        
        fs.remove_file('gdn.wpr')
        fs.add_file('data/LICENSE.txt')

        # we create another txn fs over the same location
        # to simulate an abrupt termination of the original
        # fs. this should also be a warning not to overlay
        # txn fs over the same store

        assert os.listdir('store') == []
        assert os.listdir('temp') == ['LICENSE.txt', 'rmgdn.wpr']

        #import pdb; pdb.set_trace()

        fs = TransactionalFileSystem('store', 'temp')        

        print 'temp', os.listdir('temp')
        assert os.listdir('store') == ['gdn.wpr']        
        assert os.listdir('temp') == []
    

def test_suite():
 
    return unittest.makeSuite(TransactionalFileSystemTests)

if __name__=='__main__':
    unittest.TextTestRunner().run(test_suite())    

        
