"""
 (C) Copyright 2002 Kapil Thangavelu <kvthan@wm.edu>
 All Rights Reserved

 This file is part of Gideon.

 Gideon is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 Gideon is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with Gideon; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""

# $Id: $

# File: eventchannel.py
# Author: Kapil Thangavelu <kvthan@wm.edu>
# License: GPL <www.fsf.org>
# CVS: $Id: eventchannel.py,v 1.2 2002/01/22 09:45:22 ender Exp $
# 

from OFS.SimpleItem import SimpleItem
from Globals import DTMLFile, InitializeClass
from AccessControl import ClassSecurityInfo
from App.Extensions import getObject
from Shared.DC.ZRDB.TM import TM
from Acquisition import Implicit
from ComputedAttribute import ComputedAttribute
from ZODB.PersistentMapping import PersistentMapping

from EventChannelExceptions import *
from EventFilters import validate_filter
from EventMapping import EventMappingContainer
from Presentation import EventTypePresentation, \
     EventListenerPresentation, \
     EventSourcePresentation


if 1:
    class Log:
        def debug(self,msg):pass
        def warning(self,msg):pass
        def critical(self,msg):pass
    log = Log()
    del Log
else:
    from EventLog import log

# we want to unify logging with the rest of gideon
from Products.Gideon.Log.Log import LogFactory
log = LogFactory('EventChannel')

StringType = type('')

TRANSACTION_PUBLISH_LIMIT = 10
        
manage_addEventChannelForm = DTMLFile(
    'dtml/manage_addEventChannelForm',
    globals()
    )

def manage_addEventChannel(self, id, title, REQUEST=None):
    """ oh illustrious publishing string""" 
    
    ob = EventChannel(id, title)
    self._setObject(str(id), ob)

    if REQUEST:
        self.manage_main(self, REQUEST, update_menu=1)


def dump_state(self):
    import pprint
    pprint.pprint(self._event_l_methods.items())
    pprint.pprint(self._event_types.items())
    pprint.pprint(self._event_listeners.items())
    pprint.pprint(self._event_sources.items())

def get_uid(obj):
    if type(obj) == type(''): 
        # should probably verify that infact the object exists
        # however string types uids should generally be passed 
        # in only by internal methods
        return obj
    try: uid = obj.getPhysicalPath
    except AttributeError:
        raise TypeError(
            "An event channel object must support the 'getPhysicalPath' "
            "method if no unique id is provided when cataloging"
            )
    return '/'.join(uid())

class EventChannel(SimpleItem):

    security = ClassSecurityInfo()
    
    meta_type = 'Event Channel'

    manage_options = (
                      {'label':'OverView',
                       'action':'manage_overView',
                       'help':('EventChannel', 'Overview.stx'),
                      },
                      {'label':'Event Types',
                       'action':'EventTypes/manage_main',
                       'help':('EventChannel', 'EventTypes.stx'),
                      },
                      {'label':'Listeners',
                       'action':'EventListeners/manage_main',
                       'help':('EventChannel', 'EventListeners.stx'),
                      },
                      {'label':'Sources',
                       'action':'EventSources/manage_main',
                       'help':('EventChannel', 'EventSource.stx'),
                      },
                      {'label':'Mapping',
                       'action':'EventMappings/manage_main',
                      }
                      )

    security.declareProtected('Manage Event Channel', 'manage_overView')
    manage_overView = DTMLFile('dtml/manage_overView', globals())
    
    def _v_txncontrol(self):
        v = self._v_txncontrol = EventTxnLimit()
        return v
    
    _v_txncontrol = ComputedAttribute(_v_txncontrol)
    
    def __init__(self, id, title):
        
        self.id = str(id)
        self.title = str(title)

        # helpers and presentations
        self.EventMappings = EventMappingContainer('EventMappings')
        self.EventTypes = EventTypePresentation('EventTypes')
        self.EventSources = EventSourcePresentation('EventSources')
        self.EventListeners = EventListenerPresentation('EventListeners')
        
        # may the gods of persistence
        # smile upon your efforts such that you may drink
        # of the river lethe and forget the tale of ozymandias.
        self._event_l_methods = PersistentMapping()
        self._event_types     = PersistentMapping()
        self._event_listeners = PersistentMapping()
        self._event_filters   = PersistentMapping()
        self._event_sources   = PersistentMapping()

        # configure options
        self.publishing_source_restrictions = 0

        # future options aka reminders to self
        self.ignore_invalid_registrations = 0
        self.cleanup_invalid_registrations = 0
        self.stop_publishing_on_listener_error = 0
        
        
    security.declareProtected('Use EventChannel', 'register_listener') 
    def register_listener(self,
                          object,
                          method,
                          event_type,
                          filter=None):
        
        """
        register an object for an event typewith an optional
        filter.
        """

        uid = get_uid(object)

        if type(object) is type(''):
            object = self.unrestrictedTraverse(object)

        # event type valid?
        try: listeners = self._event_types[event_type]
        except KeyError:
            raise BadEventType (
                "This event type does not exist %s"%event_type
                )
        # method valid?
        try:
            methodobj = getattr(object, method)
            assert callable(methodobj)
        except Exception, e:
            raise AttributeError (
                " event listener does not have %s method" %method
                )
        
        if listeners.count(uid) > 0:
            raise BadEventListener (
                "listener already registered for this event"
                )

        if filter is not None:
            # filter valid? raises error if not            
            validate_filter(self, object, filter)
            
            # add listener, type -> filter assoc
            self._event_filters[(uid, event_type)] = filter            
            
        # add event_type -> listener assoc
        listeners.append(uid)
        
        # trigger p machinery
        self._event_types[event_type]=listeners
        
        # add listener -> event_type assoc
        registered_et = self._event_listeners.get(uid, None)
        if registered_et:
            registered_et.append(event_type)
        else:
            self._event_listeners[uid]=[event_type]

        # add listener, type -> method assoc
        self._event_l_methods[(uid, event_type)] = method
        
    security.declareProtected('Use EventChannel', 'unregister_listener')
    def unregister_listener(self, object, event_type=None, lid=None):
        """ unregister a listener on an event channel(s),
        unless a channel is specified all registrations are
        removed """

        if lid is not None: uid = lid
        else: uid=get_uid(object)

        if not self._event_listeners.has_key(uid):
            return
        
        # remove from et listener q
        if event_type:
            if not self._event_listeners[uid].count(event_type) > 0:
                return 
            # p machinery ...
            listener_et = self._event_listeners[uid]
            listener_et.remove(event_type)

            if len(listener_et) == 0:
                del self._event_listeners[uid]
            else:
                self._event_listeners[uid]=listener_et
                
            event_types = [event_type]
        else:
            event_types = self._event_listeners[uid]
            del self._event_listeners[uid]

        # remove methods and filters
        for et in event_types:
            if self._event_filters.has_key((uid, et)):
                del self._event_filters[(uid, et)]                
            if self._event_l_methods.has_key( (uid, et) ):
                del self._event_l_methods[ (uid, et) ]
            else:
                log.warning('removed listener with no target method')

            et_listeners = self._event_types[et]
            
            if et_listeners is not None:
                et_listeners.remove(uid)
                self._event_types[et] = et_listeners

    security.declareProtected('Manage EventChannel',
                              'set_listener_filter_for_event_type')
    def set_listener_filter_for_event_type(self, object, filter, event_type):
        " "
        if not self._event_types.has_key(event_type):
            return
        if type(object) == type(''):
            object = self.unrestrictedTraverse(object)
        validate_filter(self, object, filter)
        self._event_filters[ (get_uid(object), event_type)] = filter 
        
    security.declarePublic('is_event_listener')
    def is_event_listener(self, object, event_type=None):
        """ """
        sid = get_uid(object)
        if event_type is None:
            return self._event_listeners.has_key(sid)
        elif self._event_listeners.has_key(sid) and \
             self._event_types.has_key(event_type):
            return self._event_types[event_type].count(sid) 
        return 0
    
    security.declareProtected('Use EventChannel',
                              'get_event_types_for_listener')
    def get_event_types_for_listener(self, object):
        uid = get_uid(object)
        r = self._event_listeners.get(uid, ())
        return tuple(self._event_listeners.get(uid, ()))

    security.declareProtected('Manage EventChannel',
                              'get_listener_registration')
    def get_listener_registration(self, object):

        if not self.is_event_listener(object):
            return None
        
        uid = get_uid(object)
        res = []
        
        event_types = self._event_listeners[uid]
        
        for et in event_types:
            res.append( (et,
                         self._event_l_methods.get( (uid, et) ),
                         self._event_filters.get( (uid, et), None)
                        )
                      )
        return res
    
    security.declareProtected('Manage EventChannel', 'get_event_listeners')
    def get_event_listeners(self, event_type=None):
        
        if event_type is None:
            return self._event_listeners.keys()

        listeners = self._event_types.get(event_type, ())

        return tuple(listeners)
        
        
    security.declareProtected('Use EventChannel', 'register_source')
    def register_source(self, object, event_type):
        """ add a source for a given event channel """

        uid = get_uid(object)
        log.debug('adding event source %s'%uid)        
        
        if self._event_sources.has_key(uid):
            if self._event_sources[uid].count(event_type) != 0:
                raise BadEventSource(
                    "Source already registered for this event type"
                    )
            
            self._event_sources[uid].append(event_type)
        else:
            self._event_sources[uid]=[event_type]

    security.declareProtected('Use EventChannel', 'register_source')
    def unregister_source(self, object, event_type=None):
        """ remove a source for a given event channel, if
        no channel given, remove source from all channels """
        
        uid = get_uid(object)
        log.debug('removing event source %s'%uid)
        
        if not self._event_sources.has_key(uid):
            return

        if event_type:
            self._event_sources[uid].remove(event_type)
            if len(self._event_sources[uid]) == 0:
                del self._event_sources[uid]
        else:
            del self._event_sources[uid]
            
    security.declarePublic('get_event_sources')
    def get_event_sources(self):
        return self._event_sources.keys()

    security.declareProtected('Manage Event Channel',
                              'get_source_registration')
    def get_source_registration(self, source):
        uid = get_uid(source)
        return filter(None, tuple(self._event_sources.get(uid, None)))
        
    security.declarePublic('is_source')
    def is_event_source(self, object):
        return self._event_sources.has_key(get_uid(object))
    
    security.declareProtected('Manage EventChannel', 'register_event_type') 
    def register_event_type(self, event_type):
        """ """
        if type(event_type) != type(''):
            raise BadEventType( "Event type must be a string")
        elif self._event_types.has_key(event_type):
            raise BadEventType ("%s already exists"%event_type)

        log.debug('adding event type %s'%event_type)        

        self._event_types[event_type] = []

    security.declareProtected('Manage EventChannel', 'unregister_event_type') 
    def unregister_event_type(self, event_type):
        """ """
        # unregistering an event is a fairly drastic procedure and
        # time consuming for to boot.
        
        if type(event_type) != type(''):
            raise BadEventType (
                "Event Types must be strings %s"%str(event_type)
                )
        elif not self._event_types.has_key(event_type):
            return
        
        log.debug("removing event type %s"%event_type)

        # listeners
        listeners = self._event_types.get(event_type, [])
        for l in listeners:
            self.unregister_listener(None, lid=l, event_type=event_type)

        # sources  
        sources = self._event_sources.keys()
        for s in sources:
            if self._event_sources[s].count(event_type) > 0:
                self.unregister_source(None, sid=s, event_type=event_type)

        # delete from mappings
        mappings = self.EventMappings.objectValues('Event Mapping')
        for m in mappings:
            targets = m.targets
            if event_type in targets:
                targets.remove(event_type)
                m.targets = targets

        # delete the mapping rule for it if one exists
        if event_type in self.EventMappings.objectIds():
            self.EventMappings._delObject(event_type)
        
        # event type               
        del self._event_types[event_type]
        
    security.declarePublic('has_event_type')    
    def has_event_type(self, event_type):
        return self._event_types.has_key(event_type)

    security.declareProtected('Use EventChannel', 'get_event_types')
    def get_event_types(self):
        return self._event_types.keys()

    ## mapping delegation

    security.declareProtected('Manage EventChannel', 'register_event_mapping')
    def register_event_mapping(self, source, targets):
        self.EventMappings.add_mapping(source, targets)

        
    security.declarePrivate('clear_event_channel')
    def clear_event_channel(self):
        """ clear all """
        self.__init__(self.id, self.title)

    security.declareProtected('Manage EventChannel', 'sync_registrations')
    def sync_registrations(self):
        """ remove unreachable object registrations """
        
        # should also make sure methods and filters are still
        # findable.
        
        sbozos = []
        remove = self._event_sources.__delitem__
        for s in self._event_sources.keys():
            try: sobj = self.unrestrictedTraverse(s)
            except: sbozos.append(s)
        map(remove, sbozos)

        lbozos=[]
        remove = self.unregister_listener
        for l in self._event_listeners.keys():
            try: lobj = self.unrestrictedTraverse(l)
            except: bozos.append(l)
        map(remove, lbozos)
        
        return (len(sbozos), len(lbozos))
        
    security.declareProtected('Publish Event', 'publish')
    def publish(self, obj, event_type):

        log.debug('publishing %s event_type'%event_type)
        
        if not self._event_types.has_key(event_type):
            raise BadEventType (" Event Type not registered ")
        
        if self.publishing_source_restrictions:
            
            sid = get_uid(obj)
            
            if not self._event_sources.has_key(sid):
                raise BadEventSource (
                    ' This object is not registered as an event source %s'%sid
                    )
            if not self._event_sources[sid].count(event_type) > 0:
                raise BadEventSource (
                    'This object is not registered as a source for this event'
                    )
        
        # need per transaction limit controls
        # to prevent mutation chains ie runaways
        self._v_txncontrol.check()

        # get mapping events
        event_types = [event_type]
        event_types.extend(self.EventMappings.get_mapping_types(event_type))

        for event_type in event_types:

            listeners = self._event_types[event_type]
            for l in listeners:
                try: lobj = self.unrestrictedTraverse(l)
                except:
                    log.critical(
                        "obj not found with path %s"%(l))
                    raise

                log.debug('found listener object %s'%l)
                method = self._event_l_methods.get((l,event_type), None)
                if not method:
                    log.critical(
                        "Invalid Reg: method not found with for %s %s"%(l,
                                                              event_type,
                                                              )
                        )
                    raise NoEventTargetFound (
                        "Invalid listener methodregistration entry for %s %s"%(
                        l,
                        method)
                        )

                else:
                    method = getattr(lobj, method, None)
                    if not method:
                        log.critical(
                            "method not found with for %s %s"%(l,
                                                               event_type,
                                                               )
                            )

                        raise NoEventTargetFound(
                            "no event target found for %s"%l
                            )

                try: filter = self._get_filter(lobj, l, event_type)
                except:            
                    log.critical(
                        "filter not found with name %s %s"%(l, event_type)
                        )

                    raise NoEventTargetFound (
                        "Invalid filter registration entry for %s %s"%(l,
                                                                       method
                                                                       )
                        )

                try:
                    # filters get listener, event obj, event_type
                    if not filter or filter(lobj, obj,event_type) == 1:
                        method(obj, event_type, lobj)
                except EventChannelTransactionLimit:
                    raise
                except:
                    log.debug("publish error on listener %s"%l)

                
        log.debug('done publishing')

    def _get_filter(self, listener_obj, uid, event_type):

        # add some explicit checks for filter discovery?
        
        filter_name = self._event_filters.get( (uid, event_type), None)
        
        if filter_name is None:
            return
        
        filter = getattr(listener_obj, filter_name)
        
        return filter

    security.declareProtected('Manage Event Channel', 'upgrade')
    def upgrade(self):
        """ """
        self.EventSources = EventSourcePresentation('EventSources')
        return 1

class EventTxnLimit(TM):
    
    def __init__(self):
        self._publishcount = 0
        self._registered = None # comes from TM
        
    def check(self):

        if self._publishcount < TRANSACTION_PUBLISH_LIMIT:
            self._register() 
            self._publishcount += 1            
        else:
            raise EventChannelTransactionLimit (
                     """Publishing this event would break per
                     transaction limits"""
                    )            

    def end_transaction(self):
        self._publishcount = 0

    _finish = end_transaction
    _abort  = end_transaction
    
InitializeClass(EventChannel)




