import unittest
import ZODB

from common import *
from EventChannel import EventChannel
from EventChannelExceptions import *

class EventChannelTests(unittest.TestCase):

    def setUp(self):
        self.root = BaseRoot()
        self.root.channel = EventChannel('channel', 'EventChannel')
        self.root.addObject(BasicEventListener('l1', '/l1'))
        self.root.addObject(BasicEventListener('l2', '/l2'))
        self.root.addObject(BasicEventListener('l3', '/l3'))
        self.root.addObject(BasicEventSource('s1', '/s1'))
        self.root.addObject(BasicEventSource('s2', '/s2'))
        self.root.addObject(BasicEventFilter('f1', '/f1', 1))
        self.root.addObject(BasicEventFilter('f2', '/f2', 0))
        
    def tearDown(self):
        self.root = None

    def testEventTypeRegistration(self):
        self.root.channel.register_event_type('Foo')
        self.root.channel.register_event_type('Bar')

        ets = self.root.channel.get_event_types()
        ets = list(ets)
        ets.sort()
        
        assert  ets == ['Bar', 'Foo']

    def testEventTypeUnRegistration(self):
        self.root.channel.register_event_type('Foo')
        self.root.channel.register_event_type('Bar')
        
        self.root.channel.unregister_event_type('Foo')
        self.root.channel.unregister_event_type('Bar')

        ets = self.root.channel.get_event_types()
        ets = list(ets)

        assert ets == []

    def testEventTypeUnRegistrationWithListeners(self):
        self.root.channel.register_event_type('Foo')        

        self.root.l1.f1 = BasicEventFilter('f1', '/l1/f1', 1)
        
        self.root.channel.register_listener(
            self.root.l1,
            'listen',
            'Foo',
            'f1'
            )

        self.root.channel.unregister_event_type('Foo')
        
        assert self.root.channel.get_listener_registration(self.root.l1) \
               == None

    def testEventMappingCreation(self):
        
        self.root.channel.register_event_type('Foo')
        self.root.channel.register_event_type('Bar')

        self.root.channel.EventMappings.add_mapping('Foo', ['Bar'])

        assert self.root.channel.EventMappings.Foo.targets == ['Bar']

    def testEventMappingDeletion(self):

        self.root.channel.register_event_type('Foo')
        self.root.channel.register_event_type('Bar')

        self.root.channel.EventMappings.add_mapping('Foo', ['Bar'])

        self.root.channel.EventMappings.delete_mapping('Foo')

        assert self.root.channel.EventMappings.objectIds() == []


    def testEventMappingDefaultRemoval(self):
        
        self.root.channel.register_event_type('Foo')
        self.root.channel.register_event_type('Bar')

        self.root.channel.EventMappings.add_mapping('Foo', ['Foo'])

        assert self.root.channel.EventMappings.objectIds() == []

    def testEventMappingRemovalOnTypeUnregistration(self):

        self.root.channel.register_event_type('Foo')
        self.root.channel.register_event_type('Bar')

        self.root.channel.EventMappings.add_mapping('Foo', ['Bar'])

        self.root.channel.unregister_event_type('Foo')

        assert self.root.channel.EventMappings.objectIds() == []

    def testEventListenerRegistrationWithObject(self):
        
        self.root.channel.register_event_type('Foo')
        self.root.channel.register_event_type('Bar')

        self.root.channel.register_listener(
                                                  self.root.l1,
                                                  'listen',
                                                  'Foo'
                                                  )
        assert self.root.channel.is_event_listener(self.root.l1, 'Foo') == 1

    def testEventListenerRegistrationWithPath(self):
        self.root.channel.register_event_type('Foo')
        self.root.channel.register_event_type('Bar')

        self.root.channel.register_listener(
                                                  '/l1',
                                                  'listen',
                                                  'Foo')
        
        assert self.root.channel.is_event_listener('/l1', 'Foo') == 1

    def testEventListenerRegistrationWithBadPath(self):
        self.root.channel.register_event_type('Foo')
        self.root.channel.register_event_type('Bar')

        try:
            self.root.channel.register_listener(
                                                  '/l21',
                                                  'listen',
                                                  'Foo')
        except:
            pass
        else: raise "Allowed listener registration with bad path"
        
    def testEventListenerRegistrationWithBadMethod(self):
        self.root.channel.register_event_type('Foo')        

        try:
            self.root.channel.register_listener(
                '/l1',
                'listener',
                'Foo'
                )
        except AttributeError:
            pass
        else: raise "Allowed Bad Listener Method"

    def testEventListenerRegistrationWithLocalFilter(self):

        self.root.channel.register_event_type('Foo')        

        self.root.l1.f1 = BasicEventFilter('f1', '/l1/f1', 1)
        
        self.root.channel.register_listener(
            self.root.l1,
            'listen',
            'Foo',
            'f1'
            )

        assert self.root.channel.is_event_listener(self.root.l1, 'Foo') == 1

        assert self.root.channel.get_listener_registration(self.root.l1) == \
               [('Foo', 'listen', 'f1')]
        
    def testEventListenerUnRegistrationWithFilter(self):
        
        self.root.channel.register_event_type('Foo')        

        self.root.l1.f1 = BasicEventFilter('f1', '/l1/f1', 1)
        
        self.root.channel.register_listener(
            self.root.l1,
            'listen',
            'Foo',
            'f1'
            )

        self.root.channel.unregister_listener(self.root.l1)

        assert self.root.channel.is_event_listener(self.root.l1) == 0
        assert self.root.channel._event_filters.has_key( ('/f1', 'Foo') ) == 0

    def testDuplicateListenerRegistrationsForTypeWithSameMethod(self):
        self.root.channel.register_event_type('Foo')        

        self.root.l1.f1 = BasicEventFilter('f1', '/l1/f1', 1)
        
        self.root.channel.register_listener(
            self.root.l1,
            'listen',
            'Foo',
            'f1'
            )

        try:

            self.root.channel.register_listener(
                self.root.l1,
                'listen',
                'Foo',
                'f1'
                )

        except BadEventListener:
            pass
        else: raise "Allowed duplicate registration"
            
    def testEventSourceRegistration(self):
        self.root.channel.register_event_type('Foo')        
        self.root.channel.register_source(self.root.s1, 'Foo')

        assert self.root.channel.is_event_source(self.root.s1) == 1

    def testEventSourceUnregistration(self):
        
        self.root.channel.register_event_type('Foo')        
        self.root.channel.register_source(self.root.s1, 'Foo')
        self.root.channel.unregister_source(self.root.s1)

        assert self.root.channel.is_event_source(self.root.s1) == 0                
    def testBasicEventPublishing(self):
        
        self.root.channel.register_event_type('Foo')        

        self.root.l1.f1 = BasicEventFilter('f1', '/l1/f1', 1)
        
        self.root.channel.register_listener(
            self.root.l1,
            'listen',
            'Foo',
            'f1'
            )
        self.root.channel.publish(self.root.s1, 'Foo')

        assert hasattr(self.root.l1, 'heard') == 1

    def testPublishingWithListenerException(self):

        class BadEventListener(BasicEventListener):

            def listen(self, *args): raise "foo"

        del self.root.l1 

        self.root.addObject(BadEventListener('l1', '/l1'))

        self.root.channel.register_event_type('Foo')        

        self.root.l1.f1 = BasicEventFilter('f1', '/l1/f1', 1)
        
        self.root.channel.register_listener(
            self.root.l1,
            'listen',
            'Foo',
            'f1'
            )

        try:
            self.root.channel.publish(self.root.s1, 'Foo')
        except:
            raise "Listener Exception stopped publishing"

    def testPublishingWithRemovedListener(self):

        self.root.channel.register_event_type('Foo')        

        self.root.l1.f1 = BasicEventFilter('f1', '/l1/f1', 1)
        
        self.root.channel.register_listener(
            self.root.l1,
            'listen',
            'Foo',
            'f1'
            )
        del self.root.l1

        try:
            self.root.channel.publish(self.root.s1, 'Foo')
        except:
            pass
        else: raise "Publishing with missing listener"        


    """
    def testPublishingWithRemovedFilter(self): pass

    def testPublishingWithRemovedListenerMethod(self): pass

    def testPublishingWithListenerStoppage(self): pass

    def testPublishingWithListenerStoppageAndRaise(self): pass
    """

    def testPublishingWithUnregisteredListener(self):
        
        self.root.channel.register_event_type('Foo')        

        self.root.l1.f1 = BasicEventFilter('f1', '/l1/f1', 1)
        
        self.root.channel.register_listener(
            self.root.l1,
            'listen',
            'Foo',
            'f1'
            )

        self.root.channel.unregister_listener(self.root.l1)
        
        self.root.channel.publish(self.root.s1, 'Foo')

        assert hasattr(self.root.l1, 'heard') == 0        


    def testPublishingWithEventMapping(self):

        self.root.channel.register_event_type('Foo')
        self.root.channel.register_event_type('Bar')

        self.root.channel.register_listener(
            '/l1',
            'listen',
            'Foo')

        self.root.channel.register_listener(
            '/l2',
            'listen',
            'Bar')

        self.root.channel.EventMappings.add_mapping('Foo', ['Bar'])

        self.root.channel.publish(self.root.s1, 'Foo')

        assert hasattr(self.root.l2, 'heard') == 1
    
    def testPublishingWithProxyObject(self):
        
        class EventProxy:
            def __init__(self):
                import time
                self.t = time.time()
            def get_event_time(self):
                return self.t

        event = EventProxy()

        class EventListener(BasicEventListener):

            def listen(self, event, event_type, listener):
                self.event_time = event.get_event_time()

        self.root.l4 = EventListener('l4', '/l4')
        
        self.root.channel.register_event_type('Foo')
        
        self.root.channel.register_event_type('Bar')

        self.root.channel.register_listener(
            '/l4',
            'listen',
            'Foo')

        self.root.channel.publish(event, 'Foo')

        assert self.root.l4.event_time == event.get_event_time()

    def testTxnPublishingLimits(self): 

        class HideousEventListener(BasicEventListener):
            """ cause infinite recursion """

            def listen(self, *args):
                self.getPhysicalRoot().channel.publish(self, 'Foo')

        self.root.addObject(HideousEventListener('l1', '/l1'))

        self.root.channel.register_event_type('Foo')        

        self.root.l1.f1 = BasicEventFilter('f1', '/l1/f1', 1)
        
        self.root.channel.register_listener(
            self.root.l1,
            'listen',
            'Foo',
            'f1'
            )

        try:
            self.root.channel.publish(self.root.s1, 'Foo')
        except EventChannelTransactionLimit:
            pass
        else: # if this test is broken we should get recursion limit error
            raise "Problem with transaction publishing limits"
    
def test_suite():
 
    return unittest.makeSuite(EventChannelTests)

if __name__=='__main__':
    unittest.TextTestRunner().run(test_suite())
