"""
 (C) Copyright 2002 Kapil Thangavelu <kvthan@wm.edu>
 All Rights Reserved

 this file is part of Gideon

 Gideon is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 Gideon is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with Gideon; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""

from Globals import InitializeClass, DTMLFile
from AccessControl import ClassSecurityInfo
from OFS.SimpleItem import SimpleItem

from Acquisition import aq_base

from BTrees.OOBTree import OOSet
from BTrees.Length import Length

from Products.Gideon.Objects.Application.ApplicationObject \
     import ApplicationObject

from Ratings import Rating
from RankingAlgorithm import ranking_algorithm

class RatingNotAllowed(Exception): pass

class RatingService (ApplicationObject, SimpleItem):
    
    """
    a rating tool to encapsulate object ratings.

    we use a bounded set to store the highest ranked
    objects instead of the catalog since we get
    less churn that way per vote.

    ratings are stored local to the object.
    """
    meta_type= 'Rating Service'
    security = ClassSecurityInfo()

    # after change call store.setSize
    _default_set_length = 10
    
    # don't change if you have existing ratings
    _rating_name = 'Ratings'

    highest_rated_widget = DTMLFile('ui/HighestRatedWidget', globals())
    
    def __init__(self, id):
        self.id = id
        self.store = BoundedRankSet(self._default_set_length)

    security.declarePublic('getHighestRatedObjects')
    def get_highest_rated_objects(self):
        """ return the objects with the highest ratings"""
        
        res = []
        traverse = self.unrestrictedTraverse

        for r,p in self.store.get_ranked_set():
            
            try: res.append( traverse(p) )
            except: self.store.remove_rating( r, p )
                
        return res

    security.declarePublic('getRatingForObject')
    def get_rating_for(self, content):
 
        """ return the rating for an object creating if nesc """

        if hasattr(content.aq_base, self._rating_name):
            
            return getattr(content, self._rating_name)

        elif self.is_rating_allowed_for(content):

            setattr(content, self._rating_name, Rating() )
                    
            return getattr(content, self._rating_name)
        
        else:

            raise RatingNotAllowed

            
    security.declarePublic('isRatingAllowedForObject')
    def is_rating_allowed_for(self, content):
        """ for later integration """
        return 1

    def _change_rating(self, rating):
        """ this is an internal 'callback' interface for
            ratings to talk to the tool when their updated """
        
        self.store.add_rating(rating)

InitializeClass(RatingService)
    
class BoundedRankSet(SimpleItem):

    meta_type = 'Bounded Rank Set'

    security = ClassSecurityInfo()
    security.declareObjectPrivate()

    def __init__(self, set_bound):
        
        self.ranked_set = OOSet()
        self.set_length = Length()
        self.set_bound  = set_bound

    def set_bound(self, size):

        """
        sets the bound on the number of
        ranked objects we hold.
        """

        if size < self.set_bound:

            rankings = self.ranked_set.keys()
            self.ranked_set.clear()
            self.ranked_set.update(rankings[:size])
            
        self.set_bound = size
        self.set_length.set(size)

    def add_rating(self, rating):
        """
        attempts to add a rating to the set

        if the set has not reached its bound
        then the rating is always inserted.

        otherwise the ratings ranking value
        as determined by the ranking algorithm
        function is compared to the lowest
        ranked item in the set, and is inserted
        if it has a higher value.

        """
        
        path = rating.get_content_path()
        size = rating.get_number_votes()
        value = rating.get_cumulative()
        ranking = ranking_algorithm(size, value) 

        if self.ranked_set.has_key( (ranking, path) ):
            return

        if self.set_length() <= self.set_bound:
            
            self.ranked_set.insert( (ranking, path) )
            self.set_length.change(1)
            
        else:

            key = self.ranked_set.minKey()

            if ranking > key[0]:
                
                self.ranked_set.remove(key)
                self.ranked_set.insert(
                              (ranking,  path)
                                       )
                
    def remove_rating(self, ranking, path):
        
        self.ranked_set.remove( (ranking, path) )
        self.set_length.change(-1)
        
    def get_ranked_set(self):

        set = self.ranked_set.keys()
        set.reverse()
        return set
        
InitializeClass(BoundedRankSet)        



