import unittest

from Exceptions import *
from CoreFuncs import extract_errors, extract_filters, validate

from test_utils import Request


class ValidationTests(unittest.TestCase):

	def setUp(self): pass        
	def tearDown(self): pass

	def testCanonicalCase(self):
		Req = Request({'myvar':1}, {})

		filters = {'myvar':['optional','integer'],
					'yourvar':['optional','integer']}
		
		filter_args = {}

		errors = validate(filters, filter_args, {'REQUEST':Req})

		assert len(errors) == 0, "Failed Validation Canonical Test"

	def testOptionalFilterCase(self):
		Req = Request({}, {})        
		Req2 = Request({}, {})            
		filters = {'myvar':['optional','integer'],
					'yourvar':['optional','integer']}
		
		filter_args = {}

		errors = validate(filters, filter_args, {'REQUEST':Req})

		assert len(errors) == 0, "Failed Validation Optional Test"
		assert Req == Req2, "Failed Validation Optional Test"
		
	def testFiltersWithArgsCase(self): 
		Req = Request({'foo':'1'}, {})        
		Req2 = Request({'foo':'1'}, {})
		
		filters = {'myvar':['optional','range'],
					'yourvar':['optional','integer']}
		
		filter_args = {('myvar','range'):('int','0','5')}

		errors = validate(filters, filter_args, {'REQUEST':Req})

		assert len(errors) == 0, "Failed Validation Args Test"
		assert Req == Req2, "Failed Validation Args Test"
		
	def testStrictCase(self): 
	
		Req = Request({'myvar':'1'}, {'catchme':'hahaha'})        
		Req2 = Request({'myvar':1}, {})
		
		filters = {'myvar':['optional','integer', 'range'],
					'yourvar':['optional','integer']}
		
		filter_args = {('myvar','range'):('int','0','5')}

		errors = validate(filters, filter_args, {'REQUEST':Req}, strict=1)

		assert len(errors) == 0, "Failed Validation Strict Test"
		assert Req == Req2, "Failed Validation Strict Test"

	def testErrorMessageCase(self):
		
		Req = Request({'myvar':'10'}, {})        
		Req2 = Request({'myvar':'10'}, {})
		
		filters = {'myvar':['optional','range'],
					'yourvar':['optional','integer']}
		
		filter_args = {('myvar','range'):('int','0','5')}

		errors = validate(filters, filter_args, {'REQUEST':Req})

		assert len(errors) == 1, "Failed Validation Args Test"
		assert Req == Req2, "Failed Validation Args Test"



class FilterExtractionTests(unittest.TestCase):

	def setUp(self): pass
	def tearDown(self): pass

	def testCanonicalCase(self):
		f_page = """
		myvar: integer,optional
		yourvar: integer,optional
		"""

		filters = {'myvar':['optional','integer'],
					'yourvar':['optional','integer']}

		filter_args = {}

		x, y =  extract_filters([f_page])

		assert x == filters, "Failed Canonical Case on Filters"
		assert y == filter_args, "Failed Canonical Case on Args"

	def testSpacesCase(self):

		f_page = """
		myvar : integer   ,   optional
		yourvar     : float  ,        optional
		"""

		filters = {'myvar':['optional','integer'],
					'yourvar':['optional','float']}

		filter_args = {}

		x,y= extract_filters([f_page])

		assert x == filters, "Failed Spaces Case on Filters"
		assert y == filter_args, "Failed Space Case on Args" 
		
	def testFiltersWithArgsCanonicalCase(self):

		f_page = """
		myvar:optional, range(int 20 40)

		yourvar:optional
		"""

		filters = {'myvar':['optional','range'],
					'yourvar':['optional']}

		filter_args = {('myvar','range'):('int','20','40')}

		x,y= extract_filters([f_page])

		assert x == filters, "Failed Spaces Case on Filters"
		assert y == filter_args, "Failed Space Case on Args"
		
	def testFiltersWithArgsAndSpacesCase(self):
		f_page = """
		myvar: optional, range(  int 20   40)
		"""

		filters = {'myvar':['optional', 'range']}
		filter_args = {('myvar','range'):('int','20','40')}

		x,y = extract_filters([f_page])

		assert x == filters, "Failed Spaces Case on Filters"
		assert y == filter_args, "Failed Space Case on Args"
		
	def testFiltersWithArgsAndCommasCase(self):
		f_page = """
		myvar: optional, range(int,20,40)
		"""
		try:
			x, y = extract_filters([f_page])
		except SyntaxError:
			pass
		

class ErrorExtractionTests(unittest.TestCase):

	def setUp(self): pass
	def tearDown(self): pass

	def testCanonicalCase(self):

		e_page = """
		myvar: index_html 
		yourvar: xyz_abc
		"""		 
		
		e_handles = {'myvar':'index_html','yourvar':'xyz_abc'}	
		v= extract_errors([e_page])	
		assert e_handles==v, "Error Extraction Failure, bad match"

	def testSpacesCase(self):
		
		e_page = """
		myvar : index_html
		yourvar  :  xyz_abc

		"""
		e_handles = {'myvar':'index_html','yourvar':'xyz_abc'}        
		v= extract_errors([e_page])	
		assert e_handles==v, "Error Extraction Failure, bad match"	

	def testErrorHandlerWithSpacesCase(self):
		
		e_page = """
		myvar : index html
		yourvar : xyz abc
		"""
		
		e_handles = {'myvar':'index html','yourvar':'xyz abc'}        
		v= extract_errors([e_page])
		
		assert e_handles==v, "Error Extraction Failure, bad match"

	def testMultipleColonsCase(self):
		
		e_page = """
		myvar : foo: bar
		
		"""

		try:
			v=extract_errors([e_page])
		except SyntaxError:
			pass
		else:
			raise "Failed Multiple Colons Test"
		
	def testDefaultHandlerCase(self):
		
		e_page = """
		default
		yourvar:index_html
		myvar:index_html
		"""

		e_handlers = {'__default__':'default',
						'yourvar':'index_html',
						'myvar':'index_html'}

		v = extract_errors([e_page])

		assert v == e_handlers, "Failed Default Handler Case"

	def testMultipleDefaultHandlerCase(self):

		e_page = """
		default
		yourvar:index_html
		myvar:index_html
		default2
		"""
		try:
			extract_errors([e_page])
		except SyntaxError:
			pass
		else:
			raise "Failed Multiple Colons Test"
		
		



