#
# Copyright 2001 Stfane Fermigier and Nuxeo SARL
# See LICENSE.TXT for licensing information
#

"""NuxDocument: a Zope product that represents generic documents by using
plugins to convert native productivity suite formats to HTML (for viewing or
previewing in a regular browser) or to plain text (for indexing).
"""

__version__='$Revision: 1.24 $'[11:-2]

from OFS.SimpleItem import SimpleItem, Item_w__name__
from OFS.ObjectManager import ObjectManager
from OFS.Cache import Cacheable
from OFS.History import Historical
from OFS.Image import File
import Acquisition
from DateTime.DateTime import DateTime

from AccessControl import ClassSecurityInfo
from Globals import InitializeClass

from urllib import quote
import re, string, traceback

import plugins


class NuxDocument(SimpleItem, ObjectManager):
  # (SimpleItem):
  # (CatalogAware, SimpleItem, Acquisition.Implicit, 
  # PropertyManager, Historical, Cacheable, Item_w__name__):
  # + ElementWithTitle (XXX ???)

  index_html = None # Prevent accidental acquisition (XXX ???)

  filename = _text = _html = ''
  _file = None

  # XXX: what's the right default mime type ?
  _mime_type = 'application/unknown' 

  security = ClassSecurityInfo()

  #
  # Constructor
  #
  def __init__(self, id='', title='', file='', mime_type=''):
    self._setId(id)
    self._edit(title, file, mime_type)


  # taken from CMFDefault.File
  def _isNotEmpty(self, file):
      """ Do various checks on 'file' to try to determine non emptiness. """
      if not file:
          return 0                    # Catches None, Missing.Value, ''
      elif file and (type(file) is type('')):
          return 1
      elif getattr(file, 'filename', None):
          return 1
      elif not hasattr(file, 'read'):
          return 0
      else:
          file.seek(0,2)              # 0 bytes back from end of file
          t = file.tell()             # Report the location
          file.seek(0)                # and return pointer back to 0
          if t: return 1
          else: return 0

  def _edit(self, title=None, file=None, mime_type=None):
    filename = getattr(file, 'filename', self.__name__)
    if title is not None:
      self.title = title
    notempty = self._isNotEmpty(file)
    if notempty:
      self._file = File(self.__name__, title, file)
    if mime_type:
      self._mime_type = mime_type
    elif notempty:
      self._mime_type = self._file.content_type
    self.filename = filename
    if notempty:
      self.convert()
    #self.filename = getattr(self._file, 'filename', self.__name__)
    self.publication_date = DateTime()

  def convert(self):
    file = self.getRaw()
    plugin = plugins.selectPlugin(file, self.MimeType())

    if plugin != plugins.Dumb:
      try:
        result = plugin.getConverter(self.id, file)
      except:
        traceback.print_exc()
        plugin = plugins.getDefaultPlugin()
        result = plugin.getConverter(self.id, file)
    else:
      result = plugin.getConverter(self.id, file)

    self._mime_type = result.getMimeType() or self.MimeType()
    self._text = result.getText()
    self._html = result.getBody()
    if not self.title:
      self.title = result.getTitle()
    for image_name in result.getImageNames():
      try:
        self.manage_addImage(image_name, result.getImage(image_name))
      except:
        self.manage_delObjects(ids=[image_name])
        self.manage_addImage(image_name, result.getImage(image_name))

  #
  # Accessors / rendering
  #
  security.declareProtected('View', 'getHtml')
  def getHtml(self):
    return self._html

  security.declarePrivate('getRaw')
  def getRaw(self):
    return str(self._file)
  
  security.declareProtected('View', 'MimeType')
  def MimeType(self):
    """This document's mime-type"""
    return self._mime_type

  security.declareProtected('View', 'document_src')
  def document_src(self, REQUEST=None, RESPONSE=None):
    """Unprocessed document source."""

    if RESPONSE is not None:
      RESPONSE.setHeader('Content-Type', self.MimeType())
      if self.filename:
        RESPONSE.setHeader(
          'Content-Disposition', "inline; filename=%s" % self.filename)
    return self.getRaw()

  security.declareProtected('FTP access', 'manage_FTPget')
  def manage_FTPget(self):
    "Get source for FTP download"

    return self.getRaw()

  security.declareProtected('View', 'PrincipiaSearchSource')
  def PrincipiaSearchSource(self):
    return self._text
    #return string.join(re.split(r'[^%s]+' % string.letters, self._text))

  security.declareProtected('View', 'SearchableText')
  def SearchableText(self):
    return "%s %s %s" % (
      self.title, self.Description(), self.PrincipiaSearchSource())

  security.declareProtected('Edit NuxDocument', 'PUT')
  def PUT(self, REQUEST, RESPONSE):
    "Handle HTTP/FTP PUT requests."

    self.dav__init(REQUEST, RESPONSE)
    self.html_src = REQUEST.get('BODY', '')
    self.updateState()
    self.ZCacheable_invalidate()
    self.reindex_object()
    RESPONSE.setStatus(204)
    return RESPONSE

InitializeClass(NuxDocument)

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
# 02111-1307, USA.
# 
# vim:et:ts=2:ai
