#!/usr/bin/python

#
# Copyright 2001-2002 Stfane Fermigier and Nuxeo SARL
# See LICENSE.TXT for licensing information
#
import sys, os, unittest, re, glob, string
sys.path.insert(0, '..')
import plugins

class PluginSelectionTestCase(unittest.TestCase):
  def test_selectByMimeType(self):
    self.assertEquals(
      plugins.selectPlugin(None, 'text/rtf').name, 'RTF')
    self.assertEquals(
      plugins.selectPlugin(None, 'application/msword').name, 'MSOffice')
    self.assertEquals(
      plugins.selectPlugin(None, 'application/vnd.ms-powerpoint').name, 
      'MSOffice')
    self.assertEquals(
      plugins.selectPlugin(None, 'application/vnd.ms-excel').name, 
      'MSOffice')
    self.assertEquals(
      plugins.selectPlugin(None, 'application/pdf').name, 'PDF')
    self.assertEquals(
      plugins.selectPlugin(None, 'application/vnd.sun.xml.writer').name, 
      'OpenOffice')

  def test_selectByFileType(self):
    for fn in glob.glob('test[0-9].*'):
      file = open(fn).read()
      if re.match('.*\.(ppt|xls|doc)$', fn):
        self.assertEquals(plugins.selectPlugin(file, '').name, 'MSOffice')
      elif re.match('.*\.pdf$', fn):
        self.assertEquals(plugins.selectPlugin(file, '').name, 'PDF')
      elif re.match('.*\.rtf$', fn):
        self.assertEquals(plugins.selectPlugin(file, '').name, 'RTF')
      elif re.match('.*\.(sxi|sxc|sxw)$', fn):
        self.assertEquals(plugins.selectPlugin(file, '').name, 'OpenOffice')
      else:
        self.assertEquals(plugins.selectPlugin(file, '').name, 'Dumb')


class ConverterTestCase(unittest.TestCase):

  def test_Base(self):
    doc = plugins.plugin.ConverterBase('toto', 'titi')

    self.assertEquals(doc.getRaw(), 'titi')
    self.assertEquals(doc.getHtml(), '')
    self.assertEquals(doc.getText(), '')
    self.assertEquals(doc.getImageNames(), [])
    self.assertRaises(KeyError, doc.getImage, '')

  #
  # MS-Word tests
  #
  def loadMSWordFile(self, name):
    self.result = plugins.MSOffice.getPlugin().getConverter(
      name, open('%s.doc' % name).read())

  def checkWordContentAscii(self):
    test_string = 'Linux rulez. Window$ SuCk$$$.'
    self.assertEquals(string.count(self.result.getHtml(), test_string), 1)
    self.assertEquals(string.count(self.result.getText(), test_string), 1)

  def checkWordContentUnicode(self):
    test_string = 'Il fait chaud en t.'
    self.assertEquals(string.count(self.result.getHtml(), test_string), 1)
    self.assertEquals(string.count(self.result.getText(), test_string), 1)

  def test_MSWord1(self):
    self.loadMSWordFile('test1')
    self.checkWordContentAscii()
    self.assertEquals(self.result.getTitle(), 'Untitled')

  def test_MSWord2(self):
    self.loadMSWordFile('test2')
    self.checkWordContentAscii()

  def test_MSWord3(self):
    self.loadMSWordFile('test3')
    self.checkWordContentAscii()

  def test_MSWord4(self):
    self.loadMSWordFile('test4')
    self.checkWordContentAscii()

    images_names = self.result.getImageNames()
    self.assertEquals(len(images_names), 1)
    self.assertEquals(images_names[0], 'test40.png')
    self.assertEquals(len(self.result.getImage('test40.png')), 5690)

  def test_MSWord5(self):
    self.loadMSWordFile('test5')
    self.checkWordContentUnicode()

  #
  # RTF Tests
  #
  def loadRTFFile(self, name):
    self.result = plugins.RTF.getPlugin().getConverter(
      name, open('%s.rtf' % name).read())

  def test_RTF1(self):
    self.loadRTFFile('test1')
    self.checkWordContentAscii()

  #
  # PPT Tests
  #
  def loadPPTFile(self, name):
    self.result = plugins.MSOffice.getPlugin().getConverter(
      name, open('%s.ppt' % name).read())

  def test_PPT1(self):
    self.loadPPTFile('test1')
    test_string = 'Test de ppthtml'
    self.assertEquals(string.count(self.result.getHtml(), test_string), 1)

  #
  # XLS Tests
  #
  def loadXLSFile(self, name):
    self.result = plugins.MSOffice.getPlugin().getConverter(
      name, open('%s.xls' % name).read())

  def test_XLS1(self):
    self.loadXLSFile('test1')
    test_string = 'Test de xlhtml'
    self.assertEquals(string.count(self.result.getHtml(), test_string), 1)

  #
  # OpenOffice tests
  #
  def loadOOFile(self, name, suffix):
    self.result = plugins.OpenOffice.getPlugin().getConverter(
      name, open('%s.%s' % (name, suffix)).read())

  def test_SXC1(self):
    self.loadOOFile('test1', 'sxc')
    test_string = 'Test de xlhtml'
    res = string.strip(re.sub(r'\W+', ' ', self.result.getText()))
    self.assertEquals(res, test_string)

  def test_SXI1(self):
    self.loadOOFile('test1', 'sxi')
    test_string = 'Test de ppthtml'
    res = string.strip(re.sub(r'\W+', ' ', self.result.getText()))
    self.assertEquals(res, test_string)


if __name__ == '__main__':
  unittest.main()


