#!/usr/local/bin/python
"""pyWeb Literate Programming - tangle and weave tool.

Yet another simple literate programming tool derived from nuweb, 
implemented entirely in Python.  
This produces HTML (or LATEX) for any programming language.

Usage:
    pyweb [-vs] [-c x] [-w format] [-t format] file.w

Options:
    -v           verbose output
    -s           silent output
    -c x         change the command character from '@' to x
    -w format    Use the given weaver for the final document.
                 The default is based on the input file, a leading '<'
                 indicates HTML, otherwise Latex.
    -t format    Use the given tangler to produce the output files.
    -xw          Exclude weaving
    -xt          Exclude tangling
    -pi          Permit include-command errors
    
    file.w       The input file, with @o, @d, @i, @[, @{, @|, @<, @f, @m, @u commands.
"""

__version__ = """$Revision$"""

### DO NOT EDIT THIS FILE!
### It was created by /usr/local/bin/pyweb.py, __version__='$Revision$'.
### From source pyweb.w modified Thu Jun 27 13:18:12 2002.
### In working directory '/Users/slott/Documents/Books/Python Book/pyWeb'.

import sys, os, re, time, getopt
import tempfile, filecmp



class Error( Exception ): pass


class Logger:
    def __init__( self, logConfig ):
        self.logConfig= logConfig
    def event( self, className, message ):
        className(self,message).log()

class LogActivity:
    def log( self, anEvent, aLogger ):
        pass
class LogReport( LogActivity ):
    def log( self, anEvent, aLogger ):
        print anEvent
class LogDebug( LogActivity ):
    def log( self, anEvent, aLogger ):
        print `anEvent`
class LogDiscard( LogActivity ):
    pass


class Event:
    def __init__( self, *args ):
        self.logger= args[0]
        self.args= args[1:]
        self.action= LogReport()    # default action
        self.time= time.time()
        for cl,a in self.logger.logConfig:
            if self.__class__.__name__ in cl:
                self.action= a
    def __str__( self ):
        return "%s" % ( ';'.join( self.args[:1]) )
    def __repr__( self ):
        return ("%s %s: %s" 
            % ( self.__class__.__name__, 
            time.strftime( '%c', time.localtime( self.time ) ),
            ';'.join( self.args[:1]) ) )
    def log( self ):
        self.action.log( self, self.logger )
class ExecutionEvent( Event ): 
    def __init__( self, *args ):
        apply( Event.__init__, ( self, ) + args )


class InputEvent( Event ): pass
class OptionsEvent( Event ): pass
class ReadEvent( Event ): pass
class WeaveEvent( Event ): pass
class WeaveStartEvent( Event ): pass
class WeaveEndEvent( Event ): pass
class TangleEvent( Event ): pass
class TangleStartEvent( Event ): pass
class TangleEndEvent( Event ): pass
class SummaryEvent( Event ): pass
class ErrorEvent( Event ): 
    def __str__( self ):
        return "%s" % ( ';'.join( self.args[:1]) )
class WarningEvent( ErrorEvent ): pass


_report= LogReport()
_discard= LogDiscard()
_debug= LogDebug()




class Command:
    """A Command is the lowest level of granularity in the input stream."""
    def __init__( self, fromLine=0 ):
        self.lineNumber= fromLine
    def __str__( self ):
        return "at %r" % self.lineNumber
    def startswith( self, prefix ):
        return None
    def searchForRE( self, rePat, aWeb ):
        return None
    def usedBy( self, aWeb, aChunk ):
        pass
    def weave( self, aWeb, aWeaver ):
        pass
    def tangle( self, aWeb, aTangler ):
        pass


class TextCommand( Command ):
    """A piece of document source text."""
    def __init__( self, text, fromLine=0 ):
        Command.__init__( self, fromLine )
        self.text= text
    def __str__( self ):
        return "at %r: %r..." % (self.lineNumber,self.text[:32])
    def startswith( self, prefix ):
        return self.text.startswith( prefix )
    def searchForRE( self, rePat, aWeb ):
        return rePat.search( self.text )
    def weave( self, aWeb, aWeaver ):
        aWeaver.write( self.text )
    def tangle( self, aWeb, aTangler ):
        aTangler.write( self.text )


class CodeCommand( TextCommand ):
    """A piece of program source code."""
    def weave( self, aWeb, aWeaver ):
        aWeaver.codeBlock( self.text )
    def tangle( self, aWeb, aTangler ):
        aTangler.codeBlock( self.text )


class XrefCommand( Command ):
    """Any of the Xref-goes-here commands in the input."""
    def __str__( self ):
        return "at %r: cross reference" % (self.lineNumber)
    def formatXref( self, xref, aWeaver ):
        aWeaver.xrefHead()
        xk= xref.keys()
        xk.sort()
        for n in xk:
            aWeaver.xrefLine( n, xref[n] )
        aWeaver.xrefFoot()
    def tangle( self, aWeb, aTangler ):
        raise Error('Illegal tangling of a cross reference command.')


class FileXrefCommand( XrefCommand ):
    """A FileXref command."""
    def weave( self, aWeb, aWeaver ):
        """Weave a File Xref from @o commands."""
        self.formatXref( aWeb.fileXref(), aWeaver )


class MacroXrefCommand( XrefCommand ):
    """A MacroXref command."""
    def weave( self, aWeb, aWeaver ):
        """Weave the Macro Xref from @d commands."""
        self.formatXref( aWeb.chunkXref(), aWeaver )


class UserIdXrefCommand( XrefCommand ):
    """A UserIdXref command."""
    def weave( self, aWeb, aWeaver ):
        """Weave a user identifier Xref from @d commands."""
        ux= aWeb.userNamesXref()
        aWeaver.xrefHead()
        un= ux.keys()
        un.sort()
        for u in un:
            defn, refList= ux[u]
            aWeaver.xrefDefLine( u, defn, refList )
        aWeaver.xrefFoot()


class ReferenceCommand( Command ):
    """A reference to a named chunk, via @<name@>."""
    def __init__( self, refTo, fromLine=0 ):
        Command.__init__( self, fromLine )
        self.refTo= refTo
        self.fullname= None
        self.sequenceList= None
    def __str__( self ):
        return "at %r: reference to chunk %r" % (self.lineNumber,self.refTo)
        
    def resolve( self, aWeb ):
        """Reference to actual location of this chunk"""
        self.fullName, self.sequenceList = aWeb.chunkReference( self.refTo )

        
    def usedBy( self, aWeb, aChunk ):
        self.resolve( aWeb )
        aWeb.setUsage( aChunk, self.fullName )

        
    def weave( self, aWeb, aWeaver ):
        """Create the nicely formatted reference to a chunk of code."""
        self.resolve( aWeb )
        aWeb.weaveChunk( self.fullName, aWeaver )

        
    def tangle( self, aWeb, aTangler ):
        """Create source code."""
        self.resolve( aWeb )
        aWeb.tangleChunk( self.fullName, aTangler )





class Chunk:
    """Anonymous piece of input file: will be output through the weaver only."""
    # construction and insertion into the web
    def __init__( self ):
        self.commands= [ ]
        self.lastCommand= None
        self.big_definition= None
        self.xref= None
        self.firstSecond= None
        self.name= ''
        self.seq= None
    def __str__( self ):
        return "\n".join( map( str, self.commands ) )
        
    def append( self, command ):
        """Add another Command to this chunk."""
        self.commands.append( command )

        
    def appendChar( self, text, lineNumber=0 ):
        """Append a single character to the most recent TextCommand."""
        if len(self.commands)==0 or not isinstance(self.commands[-1],TextCommand):
            self.commands.append( self.makeContent("",lineNumber) )
        self.commands[-1].text += text
        self.lastCommand= self.commands[-1]

        
    def appendText( self, text, lineNumber=0 ):
        """Add another TextCommand to this chunk or concatenate to the most recent TextCommand."""
        if self.lastCommand:
            assert len(self.commands)>=1 and isinstance(self.commands[-1],TextCommand)
            self.commands[-1].text += text
            self.lastCommand= None
        else:
            self.commands.append( self.makeContent(text,lineNumber) )

        
    def webAdd( self, web ):
        """Add self to a Web as anonymous chunk."""
        web.add( self )

    def makeContent( self, text, lineNumber=0 ):
        return TextCommand( text, lineNumber )
        
    def startswith( self, prefix ):
        """Examine the first command's starting text."""
        return len(self.commands) >= 1 and self.commands[0].startswith( prefix )
    def searchForRE( self, rePat, aWeb ):
        """Visit each command, applying the pattern."""
                
        for c in self.commands:
            if c.searchForRE( rePat, aWeb ):
                return self
        return None
    
    def usedBy( self, aWeb ):
        """Update web's used-by xref."""
                
        try:
            for t in self.commands:
                t.usedBy( aWeb, self )
        except Error,e:
            raise Error,e.args+(self,)
    
    def lineNumber( self ):
        """Return the first command's line number or None."""
        return len(self.commands) >= 1 and self.commands[0].lineNumber
    def getUserIDRefs( self ):
        return []

        
    def weave( self, aWeb, aWeaver ):
        """Create the nicely formatted document from an anonymous chunk."""
        aWeaver.docBegin( self )
        try:
            for t in self.commands:
                t.weave( aWeb, aWeaver )
        except Error, e:
            raise Error,e.args+(self,)
        aWeaver.docEnd( self )
    def weaveReferenceTo( self, aWeb, aWeaver ):
        """Create a reference to this chunk -- except for anonymous chunks."""
        raise Exception( "Cannot reference an anonymous chunk.""")
    def weaveShortReferenceTo( self, aWeb, aWeaver ):
        """Create a short reference to this chunk -- except for anonymous chunks."""
        raise Exception( "Cannot reference an anonymous chunk.""")

        
    def tangle( self, aWeb, aTangler ):
        """Create source code -- except anonymous chunks should not be tangled"""
        raise Error( 'Cannot tangle an anonymous chunk', self )



class NamedChunk( Chunk ):
    """Named piece of input file: will be output as both tangler and weaver."""
    def __init__( self, name ):
        Chunk.__init__( self )
        self.name= name
        self.seq= None
        self.fullName= None
        self.xref= []
        self.refCount= 0
    def __str__( self ):
        return "%r: %s" % ( self.name, Chunk.__str__(self) )
    def makeContent( self, text, lineNumber=0 ):
        return CodeCommand( text, lineNumber )
        
    def setUserIDRefs( self, text ):
        """Save xref variable names."""
        self.xref= text.split()
    def getUserIDRefs( self ):
        return self.xref

        
    def webAdd( self, web ):
        """Add self to a Web as named chunk, update xrefs."""
        web.addNamed( self )

        
    def weave( self, aWeb, aWeaver ):
        """Create the nicely formatted document from a chunk of code."""
        # format as <pre> in a different-colored box
        self.fullName= aWeb.fullNameFor( self.name )
        aWeaver.codeBegin( self )
        for t in self.commands:
            try:
                t.weave( aWeb, aWeaver )
            except Error,e:
                raise Error,e.args+(self,)
        aWeaver.codeEnd( self, aWeb.chunkReferencedBy( self.seq ) )
    def weaveReferenceTo( self, aWeb, aWeaver ):
        """Create a reference to this chunk."""
        self.fullName= aWeb.fullNameFor( self.name )
        aWeaver.referenceTo( self.fullName, self.seq )
    def weaveShortReferenceTo( self, aWeb, aWeaver ):
        """Create a shortened reference to this chunk."""
        aWeaver.referenceTo( None, self.seq )

        
    def tangle( self, aWeb, aTangler ):
        """Create source code."""
        # use aWeb to resolve @<namedChunk@>
        # format as correctly indented source text
        aTangler.codeBegin( self )
        for t in self.commands:
            try:
                t.tangle( aWeb, aTangler )
            except Error,e:
                raise Error,e.args+(self,)
        aTangler.codeEnd( self )



class OutputChunk( NamedChunk ):
    """Named piece of input file, defines an output tangle."""
        
    def webAdd( self, web ):
        """Add self to a Web as output chunk, update xrefs."""
        web.addOutput( self )

        
    def weave( self, aWeb, aWeaver ):
        """Create the nicely formatted document from a chunk of code."""
        # format as <pre> in a different-colored box
        self.fullName= aWeb.fullNameFor( self.name )
        aWeaver.fileBegin( self )
        try:
            for t in self.commands:
                t.weave( aWeb, aWeaver )
        except Error,e:
            raise Error,e.args+(self,)
        aWeaver.fileEnd( self, aWeb.chunkReferencedBy( self.seq ) )



class NamedDocumentChunk( NamedChunk ):
    """Named piece of input file with document source, defines an output tangle."""
    def makeContent( self, text, lineNumber=0 ):
        return TextCommand( text, lineNumber )
        
    def weave( self, aWeb, aWeaver ):
        """Ignore this when producing the document."""
        pass
    def weaveReferenceTo( self, aWeb, aWeaver ):
        """On a reference to this chunk, expand the body in place."""
        try:
            for t in self.commands:
                t.weave( aWeb, aWeaver )
        except Error,e:
            raise Error,e.args+(self,)
    def weaveShortReferenceTo( self, aWeb, aWeaver ):
        """On a reference to this chunk, expand the body in place."""
        self.weaveReferenceTo( aWeb, aWeaver )

        
    def tangle( self, aWeb, aTangler ):
        """Raise an exception on an attempt to tangle."""
        raise Error( "Cannot tangle a chunk defined with @[.""" )





class Emitter:
    """Emit an output file; handling indentation context."""
    def __init__( self ):
        self.fileName= ""
        self.theFile= None
        self.context= [0]
        self.indent= 0
        self.lastIndent= 0
        self.linesWritten= 0
        self.totalFiles= 0
        self.totalLines= 0
        
    def open( self, aFile ):
        """Open a file."""
        self.fileName= aFile
        self.doOpen( aFile )
        self.linesWritten= 0
    
    def doOpen( self, aFile ):
        self.fileName= aFile
        print "> creating %r" % self.fileName
    
    def close( self ):
        self.codeFinish()
        self.doClose()
        self.totalFiles += 1
        self.totalLines += self.linesWritten
    
    def doClose( self ):
        print "> wrote %d lines to %s" % ( self.linesWritten, self.fileName )
    
    def write( self, text ):
        self.linesWritten += text.count('\n')
        self.doWrite( text )
    
    def doWrite( self, text ):
        print text,
    

        
    def codeBlock( self, text ):
        """Indented write of a block of code."""
        self.indent= self.context[-1]
        lines= text.split( '\n' )
        for l in lines[:-1]:
            self.codeLine( '%s%s\n' % (self.indent*' ',l) )
        if lines[-1]:
            self.codeLine( '%s%s' % (self.indent*' ',lines[-1]) )
        self.lastIndent= self.indent+len(lines[-1])
    
    def codeLine( self, aLine ):
        """Each individual line of code; often overridden by weavers."""
        self.write( aLine )
    
    def codeFinish( self ):
        if self.lastIndent > 0:
            self.write('\n')

        
    def setIndent( self ):
        self.context.append( self.lastIndent )
    def clrIndent( self ):
        self.context.pop()
    def resetIndent( self ):
        self.context= [0]



class Weaver( Emitter ):
    """Format various types of XRef's and code blocks when weaving."""
        
    def doOpen( self, aFile ):
        src, junk = os.path.splitext( aFile )
        self.fileName= src + '.html'
        self.theFile= open( self.fileName, "w" )
        theLog.event( WeaveStartEvent, "Weaving %r" % self.fileName )
    def doClose( self ):
        self.theFile.close()
        theLog.event( WeaveEndEvent, "Wrote %d lines to %r" % 
            (self.linesWritten,self.fileName) )
    def doWrite( self, text ):
        self.theFile.write( text )

    # A possible Decorator interface
        
    def docBegin( self, aChunk ):
        pass
    def docEnd( self, aChunk ):
        pass

        
    def codeBegin( self, aChunk ):
        pass
    def codeEnd( self, aChunk, references ):
        pass

        
    def fileBegin( self, aChunk ):
        pass
    def fileEnd( self, aChunk, references ):
        pass

        
    def referenceTo( self, name, sequence ):
        pass

        
    def xrefHead( self ):
        pass
    def xrefFoot( self ):
        pass
    def xrefLine( self, name, refList ):
        """File Xref and Macro Xref detail line."""
        self.write( "%s: %r\n" % ( name, refList ) )
    def xrefDefLine( self, name, defn, refList ):
        """User ID Xref detail line."""
        self.write( "%s: %s, %r\n" % ( name, defn, refList ) )



class Latex( Weaver ):
    """Latex formatting for XRef's and code blocks when weaving."""
        
    def doOpen( self, aFile ):
        src, junk = os.path.splitext( aFile )
        self.fileName= src + '.tex'
        self.theFile= open( self.fileName, "w" )
        theLog.event( WeaveStartEvent, "Weaving %r" % self.fileName )

        
    def codeBegin( self, aChunk ):
        self.resetIndent()
        self.write("\\begin{flushleft} \\small")
        if not aChunk.big_definition: # defined with 'O' instead of 'o'
            self.write("\\begin{minipage}{\\linewidth}")
        self.write( " \\label{scrap%d}" % aChunk.seq )
        self.write( '\\verb@"%s"@~{\\footnotesize ' % aChunk.name )
        self.write( "\\NWtarget{nuweb%s}{%s}$\\equiv$"
            % (aChunk.name,aChunk.seq) )
        self.write( "\\vspace{-1ex}\n\\begin{list}{}{} \\item" )

        
    def codeEnd( self, aChunk, references ):
        self.write("{\\NWsep}\n\\end{list}")
        self.references( references )
        if not aChunk.big_definition: # defined with 'O' instead of 'o'
            self.write("\\end{minipage}\\\\[4ex]")
        self.write("\\end{flushleft}")

        
    def fileBegin( self, aChunk ):
        self.resetIndent()
        self.write("\\begin{flushleft} \\small")
        if not aChunk.big_definition: # defined with 'O' instead of 'o'
            self.write("\\begin{minipage}{\\linewidth}")
        self.write( " \\label{scrap%d}" % aChunk.seq )
        self.write( '\\verb@"%s"@~{\\footnotesize ' % aChunk.name )
        self.write( "\\NWtarget{nuweb%s}{%s}$\\equiv$"% (aChunk.name,aChunk.seq) )
        self.write( "\\vspace{-1ex}\n\\begin{list}{}{} \\item" )

        
    def fileEnd( self, aChunk, references ):
        self.write("{\\NWsep}\n\\end{list}")
        self.references( references )
        if not aChunk.big_definition: # defined with 'O' instead of 'o'
            self.write("\\end{minipage}\\\\[4ex]")
        self.write("\\end{flushleft}")

        
    def references( self, references ):
        if references:
            self.write("\\vspace{-1ex}")
            self.write("\\footnotesize\\addtolength{\\baselineskip}{-1ex}")
            self.write("\\begin{list}{}{\\setlength{\\itemsep}{-\\parsep}")
            self.write("\\setlength{\\itemindent}{-\\leftmargin}}")
            for n,s in references:
                self.write("\\item \\NWtxtFileDefBy\\ %s (%s)" % (n,s) )
            self.write("\\end{list}")
        else:
            self.write("\\vspace{-2ex}")

        
    def codeLine( self, aLine ):
        """Each individual line of code with LaTex decoration."""
        self.write( '\\mbox{}\\verb@%s@\\\\\n' % aLine.rstrip() )

        
    def referenceTo( self, name, sequence ):
        self.write( "\\NWlink{nuweb%s}{%s}$\\equiv$"% (name,sequence) )



class HTML( Weaver ):
    """HTML formatting for XRef's and code blocks when weaving."""
        
    def codeBegin( self, aChunk ):
        self.resetIndent()
        self.write( '\n<a name="pyweb%s"></a>\n' % ( aChunk.seq ) )
        self.write( '<!--line number %s-->' % (aChunk.lineNumber()) )
        self.write( '<p><em>%s</em> (%s)&nbsp;%s</p>\n' 
            % (aChunk.fullName,aChunk.seq,aChunk.firstSecond) )
        self.write( "<pre><code>\n" )

        
    def codeEnd( self, aChunk, references ):
        self.write( "\n</code></pre>\n" )
        self.write( '<p>&loz; <em>%s</em> (%s).' % (aChunk.fullName,aChunk.seq) )
        self.references( references )
        self.write( "</p>\n" )

        
    def fileBegin( self, aChunk ):
        self.resetIndent()
        self.write( '\n<a name="pyweb%s"></a>\n' % ( aChunk.seq ) )
        self.write( '<!--line number %s-->' % (aChunk.lineNumber()) )
        self.write( '<p><tt>"%s"</tt> (%s)&nbsp;%s</p>\n' 
            % (aChunk.fullName,aChunk.seq,aChunk.firstSecond) )
        self.write( "<pre><code>\n" )

        
    def fileEnd( self, aChunk, references ):
        self.write( "\n</code></pre>\n" )
        self.write( '<p>&loz; <tt>"%s"</tt> (%s).' % (aChunk.fullName,aChunk.seq) )
        self.references( references )
        self.write( "</p>\n" )

        
    def references( self, references ):
        if references:
            self.write( "  Used by ")
            for n,s in references:
                self.write( '<a href="#pyweb%s"><em>%s</em> (%s)</a>  ' % ( s,n,s ) )
            self.write( "." )

        
    def htmlClean( self, text ):
        """Replace basic HTML entities."""
        clean= text.replace( "&", "&amp;" ).replace( '"', "&quot;" )
        clean= clean.replace( "<", "&lt;" ).replace( ">", "&gt;" )
        return clean
    def codeLine( self, aLine ):
        """Each individual line of code with HTML cleanup."""
        self.write( self.htmlClean(aLine) )

        
    def referenceTo( self, aName, seq ):
        """Weave a reference to a chunk."""
        # Provide name to get a full reference.
        # Omit name to get a short reference.
        if aName:
            self.write( '<a href="#pyweb%s">&rarr;<em>%s</em> (%s)</a> ' 
                % ( seq, aName, seq ) )
        else:
            self.write( '<a href="#pyweb%s">(%s)</a> ' 
                % ( seq, seq ) )

        
    def xrefHead( self ):
        self.write( "<dl>\n" )
    def xrefFoot( self ):
        self.write( "</dl>\n" )
    def xrefLine( self, name, refList ):
        self.write( "<dt>%s:</dt><dd>" % name )
        for r in refList:
            self.write( '<a href="#pyweb%s">%s</a>  ' % (r,r) )
        self.write( "</dd>\n" )
    
    def xrefDefLine( self, name, defn, refList ):
        self.write( "<dt>%s:</dt><dd>" % name )
        allPlaces= refList+[defn]
        allPlaces.sort()
        for r in allPlaces:
            if r == defn:
                self.write( '<a href="#pyweb%s"><b>&bull;%s</b></a>  ' 
                    % (r,r) )
            else:
                self.write( '<a href="#pyweb%s">%s</a>  ' % (r,r) )
        self.write( "</dd>\n" )
    



class Tangler( Emitter ):
    """Tangle output files."""
        
    def doOpen( self, aFile ):
        self.fileName= aFile
        self.theFile= open( aFile, "w" )
        theLog.event( TangleStartEvent, "Tangling %r" % aFile )
    def doClose( self ):
        self.theFile.close()
        theLog.event( TangleEndEvent, "Wrote %d lines to %r" 
            % (self.linesWritten,self.fileName) )
    def doWrite( self, text ):
        self.theFile.write( text )

        
    def codeBegin( self, aChunk ):
        self.setIndent()

        
    def codeEnd( self, aChunk ):
        self.clrIndent()



class TanglerMake( Tangler ):
    """Tangle output files, leaving files untouched if there are no changes."""
    def __init__( self ):
        Tangler.__init__( self )
        self.tempname= None
        
    def doOpen( self, aFile ):
        self.tempname= tempfile.mktemp()
        self.theFile= open( self.tempname, "w" )
        theLog.event( TangleStartEvent, "Tangling %r" % aFile )

        
    def doClose( self ):
        self.theFile.close()
        try:
            same= filecmp.cmp( self.tempname, self.fileName )
        except OSError,e:
            same= 0
        if same:
            theLog.event( SummaryEvent, "No change to %r" % (self.fileName) )
            os.remove( self.tempname )
        else:
            # note the Windows requires the original file name be removed first
            os.rename( self.tempname, self.fileName )
            theLog.event( TangleEndEvent, "Wrote %d lines to %r" 
                % (self.linesWritten,self.fileName) )




class EmitterFactorySuper:
    def mkEmitter( self, name ):
        if name.lower() == 'weaver': return Weaver()
        elif name.lower() == 'tangler': return Tangler()
        return None

class EmitterFactory( EmitterFactorySuper ):
    def mkEmitter( self, name ):
        """Make an Emitter - try superclass first, then locally defined."""
        s= EmitterFactorySuper.mkEmitter( self, name )
        if s: return s
        if name.lower() == 'html': return Weaver()
        elif name.lower() == 'latex': return Latex()
        elif name.lower() == 'tanglermake': return TanglerMake()
        return None



class Web:
    """The overall Web of chunks and their cross-references."""
    def __init__( self ):
        self.sourceFileName= None
        self.chunkSeq= []
        self.output= {}
        self.named= {}
        self.usedBy= {}
        self.sequence= 0
    def __str__( self ):
        return "chunks=%r" % self.chunkSeq
        
    
    def addDefName( self, name ):
        """Reference to or definition of a chunk name."""
        nm= self.fullNameFor( name )
        if nm[-3:] == '...':
            theLog.event( ReadEvent, "Abbreviated reference %r" % name )
            return None # first occurance is a forward reference using an abbreviation
        if not self.named.has_key( nm ):
            self.named[ nm ]= []
            theLog.event( ReadEvent, "Adding chunk %r" % name )
        return nm
    
    
    def add( self, chunk ):
        """Add an anonymous chunk."""
        self.chunkSeq.append( chunk )
    
    
    def addNamed( self, chunk ):
        """Add a named chunk to a sequence with a given name."""
        self.chunkSeq.append( chunk )
        nm= self.addDefName( chunk.name )
        if nm:
            self.sequence += 1
            chunk.seq= self.sequence
            if self.named[nm]: chunk.firstSecond= '+='
            else: chunk.firstSecond= '='
            self.named[ nm ].append( chunk )
            theLog.event( ReadEvent, "Extending chunk %r from %r" % ( nm, chunk.name ) )
        else:
            raise Error("No full name for %r" % chunk.name, chunk)
    
    
    def addOutput( self, chunk ):
        """Add an output chunk to a sequence with a given name."""
        self.chunkSeq.append( chunk )
        if not self.output.has_key( chunk.name ):
            self.output[chunk.name] = []
            theLog.event( ReadEvent, "Adding chunk %r" % chunk.name )
        self.sequence += 1
        chunk.seq= self.sequence
        if self.output[chunk.name]: chunk.firstSecond= '+='
        else: chunk.firstSecond= '='
        self.output[chunk.name].append( chunk )
    

        
    def fullNameFor( self, name ):
        # resolve "..." names
        if self.named.has_key( name ): return name
        if name[-3:] == '...':
            best= []
            for n in self.named.keys():
                if n.startswith( name[:-3] ): best.append( n )
            if len(best) > 1:
                raise Error("Ambiguous abbreviation %r, matches %r" % ( name, best ) )
            elif len(best) == 1: 
                return best[0]
        return name
    
    def _chunk( self, name ):
        """Locate a named sequence of chunks."""
        nm= self.fullNameFor( name )
        if self.named.has_key( nm ):
            return self.named[nm]
        raise Error( "Cannot resolve %r in %r" % (name,self.named.keys()) )
    
    def chunkReference( self, name ):
        """Provide a full name and sequence number for a (possibly abbreviated) name."""
        c= self._chunk( name )
        if not c:
            raise Error( 'Unknown named chunk %r' % name )
        return self.fullNameFor( name ), [ x.seq for x in c ]

        
    def createUsedBy( self ):
        """Compute a "used-by" table showing references to chunks."""
        for c in self.chunkSeq:
            c.usedBy( self )
                
        for nm,cl in self.named.items():
           if len(cl) > 0:
               if cl[0].refCount == 0:
                   theLog.event( WarningEvent, "No reference to %r" % nm )
               elif cl[0].refCount > 1:
                   theLog.event( WarningEvent, "Multiple references to %r" % nm )
           else:
               theLog.event( WarningEvent, "No definition for %r" % nm )
    
    def setUsage( self, aChunk, aRefName ):
        for c in self._chunk( aRefName ):
            self.usedBy.setdefault( c.seq, [] )
            self.usedBy[c.seq].append( (self.fullNameFor(aChunk.name),aChunk.seq) )
            c.refCount += 1
    def chunkReferencedBy( self, seqNumber ):
        """Provide the list of places where a chunk is used."""
        return self.usedBy.setdefault( seqNumber, [] )
    
    def fileXref( self ):
        fx= {}
        for f,cList in self.output.items():
            fx[f]= [ c.seq for c in cList ]
        return fx
    def chunkXref( self ):
        mx= {}
        for n,cList in self.named.items():
            mx[n]= [ c.seq for c in cList ]
        return mx
    
    def userNamesXref( self ):
        ux= {}
        self._gatherUserId( self.named, ux )
        self._gatherUserId( self.output, ux )
        self._updateUserId( self.named, ux )
        self._updateUserId( self.output, ux )
        return ux
    def _gatherUserId( self, chunkMap, ux ):
                
        for n,cList in chunkMap.items():
            for c in cList:
                for id in c.getUserIDRefs():
                    ux[id]= ( c.seq, [] )
    
    def _updateUserId( self, chunkMap, ux ):
                
        # examine source for occurances of all names in ux.keys()
        for id in ux.keys():
            theLog.event( WeaveEvent, "References to %r" % id )
            idpat= re.compile( r'\W%s\W' % id )
            for n,cList in chunkMap.items():
                for c in cList:
                    if c.seq != ux[id][0] and c.searchForRE( idpat, self ):
                        ux[id][1].append( c.seq )
    

        
    def language( self, preferredWeaverClass=None ):
        """Construct a weaver appropriate to the document's language"""
        if preferredWeaverClass:
            return preferredWeaverClass()
        if self.chunkSeq[0].startswith('<'): return HTML()
        return Latex()

        
    def tangle( self, aTangler ):
        for f,c in self.output.items():
            aTangler.open( f )
            for p in c:
                p.tangle( self, aTangler )
            aTangler.close()
    def tangleChunk( self, name, aTangler ):
        theLog.event( TangleEvent, "Tangling chunk %r" % name )
        for p in self._chunk(name):
            p.tangle( self, aTangler )

        
    def weave( self, aWeaver ):
        aWeaver.open( self.sourceFileName )
        for c in self.chunkSeq:
            c.weave( self, aWeaver )
        aWeaver.close()
    def weaveChunk( self, name, aWeaver ):
        theLog.event( WeaveEvent, "Weaving chunk %r" % name )
        chunkList= self._chunk(name)
        chunkList[0].weaveReferenceTo( self, aWeaver )
        for p in chunkList[1:]:
            p.weaveShortReferenceTo( self, aWeaver )



class WebReader:
    """Parse an input file, creating Commands and Chunks."""
    def __init__( self, fileName, parent=None, command='@', permit=() ):
        self.fileName= fileName
        self.tokenList= []
        self.token= ""
        self.tokenIndex= 0
        self.tokenPushback= []
        self.lineNumber= 0
        self.aChunk= None
        self.parent= parent
        self.theWeb= None
        self.permitList= permit
        if self.parent: 
            self.command= self.parent.command
        else:
            self.command= command
        self.parsePat= '(%s.)' % self.command
                
        # major commands
        self.cmdo= self.command+'o'
        self.cmdo_big= self.command+'O'
        self.cmdd= self.command+'d'
        self.cmdlcurl= self.command+'{'
        self.cmdrcurl= self.command+'}'
        self.cmdlbrak= self.command+'['
        self.cmdrbrak= self.command+']'
        self.cmdi= self.command+'i'
        # minor commands
        self.cmdlangl= self.command+'<'
        self.cmdrangl= self.command+'>'
        self.cmdpipe= self.command+'|'
        self.cmdlexpr= self.command+'('
        self.cmdrexpr= self.command+')'
        self.cmdf= self.command+'f'
        self.cmdm= self.command+'m'
        self.cmdu= self.command+'u'
        self.cmdcmd= self.command+self.command

        
    def openSource( self ):
        theLog.event( InputEvent, "Processing %r" % self.fileName )
        file= open(self.fileName, 'r' ).read()
        self.tokenList= re.split(self.parsePat, file )
        self.lineNumber= 1
        self.tokenPushback= []
    def nextToken( self ):
        self.lineNumber += self.token.count('\n')
        if self.tokenPushback:
            self.token= self.tokenPushback.pop()
        else:
            self.token= self.tokenList.pop(0)
        return self.token
    def moreTokens( self ):
        return self.tokenList or self.tokenPushback
    def pushBack( self, token ):
        self.tokenPushback.append( token )
    def totalLines( self ):
        self.lineNumber += self.token.count('\n')
        return self.lineNumber-1

        
    def location( self ):
        return ( self.fileName, self.lineNumber, self.lineNumber+self.token.count("\n") )

        
    def handleCommand( self, token ):
                
        if token[:2] == self.cmdo or token[:2] == self.cmdo_big:
                        
            file= self.nextToken().strip()
            self.aChunk= OutputChunk( file )
            self.aChunk.webAdd( self.theWeb )
            self.aChunk.big_definition= token[:2] == self.cmdo_big
            self.expect( (self.cmdlcurl,) )
            # capture an OutputChunk up to @}
        
        elif token[:2] == self.cmdd:
                        
            name= self.nextToken().strip()
            # next token is @{ or @]
            brack= self.expect( (self.cmdlcurl,self.cmdlbrak) )
            if brack == self.cmdlcurl: 
                self.aChunk= NamedChunk( name )
            else: 
                self.aChunk= NamedDocumentChunk( name )
            self.aChunk.webAdd( self.theWeb )
            # capture a NamedChunk up to @} or @]
        
        elif token[:2] == self.cmdi:
                        
            # break this token on the '\n' and pushback the new token.
            next= self.nextToken().split('\n',1)
            self.pushBack('\n')
            if len(next) > 1:
                self.pushBack( '\n'.join(next[1:]) )
            incFile= next[0].strip()
            try:
                include= WebReader( incFile, parent=self )
                include.load( self.theWeb )
            except (Error,IOError),e:
                theLog.event( ErrorEvent, 
                    "Problems with included file %s, output is incomplete." 
                    % incFile )
                # Discretionary - sometimes we want total failure
                if self.cmdi in self.permitList: pass
                else: raise
            self.aChunk= Chunk()
            self.aChunk.webAdd( self.theWeb )
        
        elif token[:2] in (self.cmdrcurl,self.cmdrbrak):
                        
            self.aChunk= Chunk()
            self.aChunk.webAdd( self.theWeb )
        
    
                
        elif token[:2] == self.cmdpipe:
                        
            # variable references at the end of a NamedChunk
            # aChunk must be subclass of NamedChunk
            # These are accumulated and expanded by @u reference
            self.aChunk.setUserIDRefs( self.nextToken().strip() )
        
        elif token[:2] == self.cmdf:
            self.aChunk.append( FileXrefCommand(self.lineNumber) )
        elif token[:2] == self.cmdm:
            self.aChunk.append( MacroXrefCommand(self.lineNumber) )
        elif token[:2] == self.cmdu:
            self.aChunk.append( UserIdXrefCommand(self.lineNumber) )
        elif token[:2] == self.cmdlangl:
                        
            # get the name, introduce into the named Chunk dictionary
            expand= self.nextToken().strip()
            self.expect( (self.cmdrangl,) )
            self.theWeb.addDefName( expand )
            self.aChunk.append( ReferenceCommand( expand, self.lineNumber ) )
        
        elif token[:2] == self.cmdlexpr:
                        
            # get the Python expression, create the expression command
            expression= self.nextToken()
            self.expect( (self.cmdrexpr,) )
            try:
                theLocation= self.location()
                theWebReader= self
                theFile= self.fileName
                thisApplication= sys.argv[0]
                result= str(eval( expression ))
            except Exception,e:
                result= '!!!Exception: %s' % e
                theLog.event( ReadEvent, 'Failure to process %r: result is %s' % ( expression, e ) )
            self.aChunk.appendText( result, self.lineNumber )
        
        elif token[:2] == self.cmdcmd:
                        
            # replace with '@' here and now!
            # Put this at the end of the previous chunk
            # AND make sure the next chunk is appended to this.
            self.aChunk.appendChar( self.command, self.lineNumber )
        
    
        elif token[:2] in (self.cmdlcurl,self.cmdlbrak):
            # These should be consumed as part of @o and @d parsing
            raise Error('Extra %r (possibly missing chunk name)' % token, self.aChunk)
        else:
            return None # did not recogize the command
        return 1 # did recognize the command

        
    def expect( self, tokens ):
        if not self.moreTokens():
            raise Error("At %r: end of input, %r not found" % (self.location(),tokens) )
        t= self.nextToken()
        if t not in tokens:
            raise Error("At %r: expected %r, found %r" % (self.location(),tokens,t) )
        return t
    def load( self, aWeb ):
        self.theWeb= aWeb
        self.aChunk= Chunk()
        self.aChunk.webAdd( self.theWeb )
        self.openSource()
        while self.moreTokens():
            token= self.nextToken()
            if len(token) >= 2 and token.startswith(self.command):
                if self.handleCommand( token ):
                    continue
                else:
                                        
                    theLog.event( ReadEvent, 'Unknown @-command in input: %r' % token )
                    self.aChunk.appendText( token, self.lineNumber )
    
            elif token:
                # accumulate non-empty block of text in the current chunk
                self.aChunk.appendText( token, self.lineNumber )




class Operation:
    """An operation performed by pyWeb."""
    def __init__( self, name ):
        self.name= name
        self.start= None
    def __str__( self ):
        return self.name
        
    def perform( self, theWeb, theApplication ):
        theLog.event( ExecutionEvent, "Starting %s" % self.name )
        self.start= time.clock()

        
    def duration( self ):
        """Return duration of the operation."""
        # Windows clock() function is funny.
        return (self.start and time.clock()-self.start) or 0
    def summary( self, theWeb, theApplication ):
        return "%s in %0.1f sec." % ( self.name, self.duration() )



class MacroOperation( Operation ):
    """An operation composed of a sequence of other operations."""
    def __init__( self, opSequence=None ):
        Operation.__init__( self, "Macro" )
        if opSequence: self.opSequence= opSequence
        else: self.opSequence= []
    def __str__( self ):
        return "; ".join( [ x.name for x in self.opSequence ] )
        
    def perform( self, theWeb, theApplication ):
        for o in self.opSequence:
            o.perform(theWeb,theApplication)

        
    def append( self, anOperation ):
        self.opSequence.append( anOperation )

        
    def summary( self, theWeb, theApplication ):
        return ", ".join( [ x.summary(theWeb,theApplication) for x in self.opSequence ] )



class WeaveOperation( Operation ):
    """An operation that weaves a document."""
    def __init__( self ):
        Operation.__init__( self, "Weave" )
        
    def perform( self, theWeb, theApplication ):
        Operation.perform( self, theWeb, theApplication )
        if not theApplication.theWeaver: 
            # Examine first few chars of first chunk of web to determine language
            theApplication.theWeaver= theWeb.language() 
        try:
            theWeb.weave( theApplication.theWeaver )
        except Error,e:
            theLog.event( ErrorEvent, 
                "Problems weaving document from %s (weave file is faulty)." 
                % theWeb.sourceFileName )
            raise

        
    def summary( self, theWeb, theApplication ):
        if theApplication.theWeaver and theApplication.theWeaver.linesWritten > 0:
            return "%s %d lines in %0.1f sec." % ( self.name, theApplication.theWeaver.linesWritten, self.duration() )
        return "did not %s" % ( self.name, )



class TangleOperation( Operation ):
    """An operation that weaves a document."""
    def __init__( self ):
        Operation.__init__( self, "Tangle" )
        
    def perform( self, theWeb, theApplication ):
        Operation.perform( self, theWeb, theApplication )
        try:
            theWeb.tangle( theApplication.theTangler )
        except Error,e:
            theLog.event( ErrorEvent, 
                "Problems tangling outputs from %s (tangle files are faulty)." 
                % theWeb.sourceFileName )
            raise

        
    def summary( self, theWeb, theApplication ):
        if theApplication.theTangler and theApplication.theTangler.linesWritten > 0:
            return "%s %d lines in %0.1f sec." % ( self.name, theApplication.theTangler.linesWritten, self.duration() )
        return "did not %s" % ( self.name, )



class LoadOperation( Operation ):
    """An operation that loads the source web for a document."""
    def __init__( self ):
        Operation.__init__( self, "Load" )
        
    def perform( self, theWeb, theApplication ):
        Operation.perform( self, theWeb, theApplication )
        try:
            theApplication.webReader.load( theWeb )
            theWeb.createUsedBy()
        except (Error,IOError),e:
            theLog.event( ErrorEvent, 
                "Problems with source file %s, no output produced." 
                % theWeb.sourceFileName )
            raise

        
    def summary( self, theWeb, theApplication ):
        return "%s %d lines in %01.f sec." % ( self.name, theApplication.webReader.totalLines(), self.duration() )





class Application:
    def __init__( self, ef=None ):
                
        if not ef: ef= EmitterFactory()
        self.emitterFact= ef
        self.theTangler= ef.mkEmitter( 'TanglerMake' )
        self.theWeaver= None
        self.commandChar= '@'
        loadOp= LoadOperation()
        weaveOp= WeaveOperation()
        tangleOp= TangleOperation()
        self.doWeave= MacroOperation( [loadOp,weaveOp] )
        self.doTangle= MacroOperation( [loadOp,tangleOp] )
        self.theOperation= MacroOperation( [loadOp,tangleOp,weaveOp] )
        self.permitList= []
        self.files= []
        self.webReader= None

        
    def parseArgs( self, argv ):
        global theLog
        opts, self.files = getopt.getopt( argv[1:], 'hvsdc:w:t:x:p:' )
        for o,v in opts:
            if o == '-c': 
                theLog.event( OptionsEvent, "Setting command character to %r" % v )
                self.commandChar= v
            elif o == '-w': 
                theLog.event( OptionsEvent, "Setting weaver to %r" % v )
                self.theWeaver= self.emitterFact.mkEmitter( v )
            elif o == '-t': 
                theLog.event( OptionsEvent, "Setting tangler to %r" % v )
                self.theTangler= self.emitterFact.mkEmitter( v )
            elif o == '-x':
                if v.lower().startswith('w'): # skip weaving
                    self.theOperation= self.doTangle
                elif v.lower().startswith('t'): # skip tangling
                    self.theOperation= self.doWeave
                else:
                    raise Exception( "Unknown -x option %r" % v )
            elif o == '-p':
                # save permitted errors, usual case is -pi to permit include errors
                self.permitList= [ '%s%s' % ( commandChar, c ) for c in v ]
            elif o == '-h': print __doc__
            elif o == '-v': theLog= Logger( verbose )
            elif o == '-s': theLog= Logger( silent )
            elif o == '-d': theLog= Logger( debug )
            else:
                raise Exception('unknown option %r' % o)

        
    def process( self, theFiles=None ):
        for f in theFiles or self.files:
            self.webReader= WebReader( f, command=self.commandChar, permit=self.permitList )
            try:
                w= Web()
                w.sourceFileName= self.webReader.fileName
                self.theOperation.perform( w, self )
            except Error,e:
                print '>', e.args[0]
                for a in e.args[1:]:
                    print '...', a
            except IOError,e:
                print e
            theLog.event(SummaryEvent, 'pyWeb: %s' % self.theOperation.summary(w,self) )



standard = [
    (('ErrorEvent','WarningEvent'), _report ),
    (('ExecutionEvent',), _debug ),
    (('InputEvent','WeaveStartEvent', 'WeaveEndEvent', 
      'TangleStartEvent', 'TangleEndEvent','SummaryEvent'), _report),
    (('OptionsEvent', 'ReadEvent', 
      'WeaveEvent', 'TangleEvent'),  _discard ) ]
silent = [
    (('ErrorEvent','SummaryEvent'), _report ),
    (('ExecutionEvent','WarningEvent',
      'InputEvent','WeaveStartEvent', 'WeaveEndEvent', 
      'TangleStartEvent', 'TangleEndEvent'), _discard),
    (('Optionsevent', 'ReadEvent', 
      'WeaveEvent', 'TangleEvent'),  _discard ) ]
verbose = [
    (('ErrorEvent','WarningEvent'), _report ),
    (('ExecutionEvent',
      'InputEvent','WeaveStartEvent', 'WeaveEndEvent', 
      'TangleStartEvent', 'TangleEndEvent','SummaryEvent'), _report),
    (('OptionsEvent', 'ReadEvent',
      'WeaveEvent', 'TangleEvent'),  _report ) ]

theLog= Logger(standard)


def main( ef, argv ):
    a= Application( ef )
    a.parseArgs( argv )
    a.process()

if __name__ == "__main__":
    main( EmitterFactory(), sys.argv )

def tangle( aFile ):
    """Tangle a single file, permitting errors on missing @i files."""
    a= Application( EmitterFactory() )
    a.permitList= [ '%si' % a.commandChar ]
    a.theOperation= a.doTangle
    a.process( [aFile] )
    
def weave( aFile ):
    """Weave a single file."""
    a= Application( EmitterFactory() )
    a.theOperation= a.doWeave
    a.process( [aFile] )

