__doc__ = '''Calendar tag'''
__version__='1.0.17'


from string import capitalize, join, replace, split
from types import StringType


from DocumentTemplate.DT_Util import Eval, InstanceDict, ParseError, \
                                     namespace, parse_params, render_blocks
from DocumentTemplate.DT_String import String
from DateTime.DateTime import DateTime

import re

try:
    # For backwards compatibility
    from DocumentTemplate.DT_Util import expr_globals

    old_Eval = Eval
    def Eval(expr):
        return old_Eval(expr, expr_globals)
except ImportError:
    # Not needed since Zope 2.4
    pass


##from time import time

start = """
<!-- Calendar Begin -->
<style>
  .calhead {
            font-family:verdana,sans-serif;
            font-size:12 px;
           }
  .calendar {
            font-size:1.2em;
            border: solid rgb(0,0,0);
            border-width: 1px 1px 1px 1px;
            margin: 0px;
            padding: 3px;
            text-decoration:none;
           }
  .caldate {
            font-size:1.2em;
            margin: 0px;
            padding: 3px;
            text-decoration:none;
           }
</style>
<table 
       border="0" bgcolor="%(bgcolor)s" cellspacing="0" cellpadding="0"%(tablewidth)s>
 <tr>
    <td colspan="1">
      <!-- Toolbar -->
      <table border="0" cellspacing="0" cellpadding="0">
      <tr>
          <td align="center" valign="middle" bgcolor="%(leftbgcolor)s">
              <font color="%(leftfgcolor)s" class="calhead"><b>%(left)s</b></font>
          </td>
          %(middle)s
          %(right)s
     </tr>
     </table>
    </td>
 </tr>
 %(start_optional)s
"""

end = """
</table>
<!-- Calendar End -->
"""

body_start = """
 <tr>
  <td bgcolor="%(bordercolor)s" align="center" valign="middle" colspan="%(colspan)s">
   <table  
            border="0" width="100%%" cellspacing="%(spacing)s" cellpadding="%(padding)s">
"""

body_end = """
   </table>
  </td>
 </tr>
"""

right = """
  <td align="right" valign="middle" bgcolor="%(rightbgcolor)s">
      <a href="%(prev_url_)s">%(prev_link_)s</a>%(rightmiddle)s<a href="%(next_url_)s">%(next_link_)s</a>
  </td>
"""

## Translate machinery
daynames = {'ca': ('Du', 'Dll', 'Dm', 'Dx', 'Dj', 'Dv', 'Ds'),
            'da': ('Sn','Man','Tir','Ons','Tor','Fre','Lr'),
            'de': ('So', 'Mo', 'Di', 'Mi', 'Do', 'Fr', 'Sa'),
            'du': ('Zo', 'M', 'Di', 'W', 'Do', 'V', 'Za'),
            'en': 'SMTWTFS',
            'es': ('D', 'L', 'M', 'Mi', 'J', 'V', 'S'),
            'eu': ('Ig','Al', 'Aa', 'Az', 'Og', 'Or', 'La'),
            'fi': ('Su', 'Ma', 'Ti', 'Ke', 'To', 'Pe', 'La'),
            'fr': ('D', 'L', 'M', 'M', 'J', 'V', 'S'),
            'it': ('Do', 'Lu', 'Ma', 'Me', 'Gio', 'Ve', 'Sa'),
            'ja': ('', '', '', '', '', '', ''),
            'no': 'SMTOTFL',
            'pl': ('N', 'Pn', 'Wt', 'r', 'Cz', 'Pt', 'So'),
            'pt': ('Dom', 'Seg', 'Ter', 'Qua', 'Qui', 'Sex', 'Sb'),
            'ro': ('D', 'L', 'M', 'M', 'J', 'V', 'S'),
            'ru': ('', '', '', '', '', '', ''),
	    'zh_TW': ('g', 'g@', 'gG', 'gT', 'g|', 'g', 'g')
            }

monthnames = {'ca': ('Gener','Febrer','Mar&ccedil;','Abril','Maig','Juny','Juliol','Agost','Setembre','Octubre','Novembre','Desembre'),
            'da': ('Januar','Februar','Marts','April','Maj','Juni','Juli','August','September','Oktober','November','December'),
            'de': ('Januar','Februar','M&auml;rz','April','Mai','Juni','Juli','August','September','Oktober','November','Dezember'),
            'du': ('Januari','Februari','Maart','April','Mei','Juni','Juli','Augustus','September','October','November','December'),
            'en': ('January','February','March','April','May','June','July','August', 'September','October','November','December'),
            'es': ('Enero','Febrero','Marzo','Abril','Mayo','Junio','Julio','Agosto', 'Septiembre','Octubre','Noviembre','Diciembre'),
            'eu': ('Urtarrila','Otsaila','Martxoa','Apirila','Maiatza','Ekaina','Uztaila','Abuztua', 'Iraila','Urria','Azaroa','Abendua'),
            'fi': ('Tammikuu','Helmikuu','Maaliskuu','Huhtikuu','Toukokuu','Keskuu','Heinkuu','Elokuu','Syyskuu','Lokakuu','Marraskuu','Joulukuu'),
            'fr': ('Janvier','F&eacute;vrier','Mars','Avril','Mai','Juin','Juillet','Ao&ucirc;t','Septembre','Octobre','Novembre','D&eacute;cembre'),
            'it': ('Gennaio','Febbraio','Marzo','Aprile','Maggio', 'Giugno','Luglio','Agosto','Settembre','Ottobre','Novembre','Dicembre'),
            'ja': ('1','2','3','4','5','6','7','8','9','10','11','12'),
            'no': ('Januar','Februar','Mars','April','Mai','Juni','Juli','August','September','Oktober','November','Desember'),
            'pl': ('Stycze','Luty','Marzec','Kwiecie','Maj','Czerwiec','Lipiec','Sierpie','Wrzesie','Padziernik','Listopad','Grudzie'),
            'pt': ('Janeiro','Fevereiro','Maro','Abril','Maio','Junho','Julho','Agosto','Setembro','Outubro','Novembro','Dezembro'),
            'ro': ('Ianuarie','Februarie','Martie','Aprilie','Mai','Iunie','Iulie','August','Septembrie','Octombrie','Novembrie','Decembrie'),
            'ru': ('','','','','','','','','','','',''),
            'zh_TW': ('@','G','T','|','','','C','K','E','Q','Q@','QG')
	    }

shortmonthnames = {'ca': ('Gen','Feb','Mar','Abr','Mai','Jun','Jul','Ago','Set','Oct','Nov','Des'),
            'da': ('Jan','Feb','Mar','Apr','Maj','Jun','Jul','Aug','Sep','Okt','Nov','Dec'),
            'de': ('Jan','Feb','M&auml;r','Apr','Mai','Jun','Jul','Aug','Sep','Okt','Nov','Dez'),
            'du': ('Jan','Feb','Maa','Apr','Mei','Jun','Jul','Aug','Sep','Oct','Nov','Dec'),
            'en': ('Jan','Feb','Mar','Apr','May','Jun','Jul','Aug', 'Sep','Oct','Nov','Dec'),
            'es': ('Ene','Feb','Mar','Abr','May','Jun','Jul','Ago', 'Sep','Oct','Nov','Dic'),
            'eu': ('Urt','Ots','Mar','Api','Mai','Eka','Uzt','Abu', 'Ira','Urr','Aza','Abe'),
            'fi': ('Tam','Hel','Maa','Huh','Tou','Kes','Hei','Elo','Syy','Lok','Mar','Jou'),
            'fr': ('Jan','F&eacute;v','Mar','Avr','Mai','Juin','Juil','Ao&ucirc;','Sep','Oct','Nov','D&eacute;c'),
            'it': ('Gen','Feb','Mar','Apr','Mag', 'Giu','Lug','Ago','Set','Ott','Nov','Dic'),
            'ja': ('1','2','3','4','5','6','7','8','9','10','11','12'),
            'no': ('Jan','Feb','Mar','Apr','Mai','Jun','Jul','Aug','Sep','Okt','Nov','Des'),
            'pl': ('Sty','Lut','Mar','Kwi','Maj','Cze','Lip','Sie','Wrz','Pa','Lis','Gru'),
            'pt': ('Jan','Fev','Mar','Abr','Mai','Jun','Jul','Ago','Set','Out','Nov','Dez'),
            'ro': ('Ian','Feb','Mar','Apr','Mai','Iun','Iul','Aug','Sep','Oct','Nov','Dec'),
            'ru': ('','','','','','','','','','','',''),
	    'zh_TW': ('@','G','T','|','','','C','K','E','Q','Q@','QG')
	    }

translations = {
    'Day'  : {'ca': 'Dia',
              'da': 'Dag',
              'de': 'Tag',
              'du': 'Dag',
              'es': 'D&iacute;a',
              'eu': 'Eguna',
              'fi': 'Piv',
              'fr': 'Jour',
              'it': 'Giorno',
              'ja': '',
              'no': 'Dag',
              'pl': 'Dzie',
              'pt': 'Dia',
              'ro': 'Zi',
              'ru': '',
              'zh_TW': '@'},
    'Week' : {'ca': 'Setmana',
              'da': 'Uge',
              'de': 'Woche',
              'du': 'Week',
              'es': 'Semana',
              'eu': 'Astea',
              'fi': 'Viikko',
              'fr': 'Semaine',
              'it': 'Settimana',
              'ja': '',
              'no': 'Uke',
              'pl': 'Tydzie',
              'pt': 'Semana',
              'ro': 'Saptam&icirc;na',
              'ru': '',
              'zh_TW': '@g'},
    'Month': {'ca': 'Mes',
              'da': 'M&aring;ned',
              'de': 'Monat',
              'du': 'Maand',
              'es': 'Mes',
              'eu': 'Hilabetea',
              'fi': 'Kuukausi',
              'fr': 'Mois',
              'it': 'Mese',
              'ja': '',
              'no': 'M&aring;ned',
              'pl': 'Miesic',
              'pt': 'Ms',
              'ro': 'Luna',
              'ru': '',
              'zh_TW': '@'},
    'Year' : {'ca': 'Any',
              'da': '&Aring;r',
              'de': 'Jahr',
              'du': 'Jaar',
              'es': 'Ao',
              'eu': 'Urtea',
              'fi': 'Vuosi',
              'fr': 'Ann&eacute;e',
              'it': 'Anno',
              'ja': 'ǯ',
              'no': '&Aring;r',
              'pl': 'Rok',
              'pt': 'Ano',
              'ro': 'Anul',
              'ru': '',
              'zh_TW': '@~'},
    'Previous Week' : {'ca': 'Setmana previa',
                       'da': 'Forrige Uge',
                       'de': 'Vorige Woche',
                       'du': 'Vorige Week',
                       'es': 'Semana previa',
                       'eu': 'Aurreko astea',
                       'fi': 'Edellinen viikko',
                       'fr': 'Semaine Pr&eacute;c&eacute;dente',
                       'it': 'Settimana precedente',
                       'ja': '轵',
                       'no': 'Forrige Uke',
                       'pl': 'Poprzedni Tydzie',
                       'pt': 'Semana Anterior',
                       'ro': 'Saptam&icirc;na trecuta',
                       'ru': ' ',
                       'zh_TW': 'W@g'},
    'Next Week'     : {'ca': 'Setmana seg&uuml;ent',
                       'da': 'Nste uge',
                       'de': 'N&auml;chste Woche',
                       'du': 'Volgende Week',
                       'es': 'Semana siguiente',
                       'eu': 'Ondoko astea',
                       'fi': 'Seuraava viikko',
                       'fr': 'Semaine Suivante',
                       'it': 'Settimana successiva',
                       'ja': '轵',
                       'no': 'Neste Uke',
                       'pl': 'Nastpny Tydzie',
                       'pt': 'Prxima Semana',
                       'ro': 'Saptam&icirc;na viitoare',
                       'ru': ' ',
                       'zh_TW': 'U@g'},
    'Previous Month': {'ca': 'Mes previ',
                       'da': 'Forrige m&aring;ned',
                       'de': 'Voriger Monat',
                       'du': 'Vorige Maand',
                       'es': 'Mes previo',
                       'eu': 'Aurreko hilabetea',
                       'fi': 'Edellinen kuukausi',
                       'fr': 'Mois Pr&eacute;c&eacute;dent',
                       'it': 'Mese precedente',
                       'ja': '',
                       'no': 'Forrige M&aring;ned',
                       'pl': 'Poprzedni Miesic',
                       'pt': 'Ms Anterior',
                       'ro': 'Luna trecuta',
                       'ru': ' ',
                       'zh_TW': 'WӤ'},
    'Next Month'    : {'ca': 'Mes seg&uuml;ent',
                       'da': 'Nste m&aring;ned',
                       'de': 'N&auml;chster Monat',
                       'du': 'Volgende Maand',
                       'es': 'Mes siguiente',
                       'eu': 'Ondoko hilabetea',
                       'fi': 'Seuraava kuukausi',
                       'fr': 'Mois Suivant',
                       'it': 'Mese successivo',
                       'ja': '',
                       'no': 'Neste M&aring;ned',
                       'pl': 'Nastpny Miesic',
                       'pt': 'Prximo Ms',
                       'ro': 'Luna viitoare',
                       'ru': ' ',
                       'zh_TW': 'UӤ'},
    'Previous Year' : {'ca': 'Any previ',
                       'da': 'Forrige &aring;',
                       'de': 'Voriges Jahr',
                       'du': 'Vorig Jaar',
                       'es': 'A&ntilde;o previo',
                       'eu': 'Aurreko urtea',
                       'fi': 'Edellinen vuosi',
                       'fr': 'Ann&eacute;e Pr&eacute;c&eacute;dente',
                       'it': 'Anno precedente',
                       'ja': 'ǯ',
                       'no': 'Forrige &Aring;r',
                       'pl': 'Poprzedni Rok',
                       'pt': 'Ano Anterior',
                       'ro': 'Anul trecut',
                       'ru': ' ',
                       'zh_TW': 'W@~'},
    'Next Year'     : {'ca': 'Any seg&uuml;ent',
                       'da': 'Nste &Aring;',
                       'de': 'N&auml;chstes Jahr',
                       'du': 'Volgend Jaar',
                       'es': 'A&ntilde;o siguiente',
                       'eu': 'Hurrengo urtea',
                       'fi': 'Seuraava vuosi',
                       'fr': 'Ann&eacute;e Suivante',
                       'it': 'Anno successivo',
                       'ja': 'ǯ',
                       'no': 'Neste &Aring;r',
                       'pl': 'Nastpny Rok',
                       'pt': 'Prximo Ano',
                       'ro': 'Anul viitor',
                       'ru': ' ',
                       'zh_TW': 'U@~'},
##    'January'  : {'ca': 'Gener',
##                  'da': 'Januar',
##                  'de': 'Januar',
##                  'du': 'Januari',
##                  'es': 'Enero',
##                  'fi': 'Tammikuu',
##                  'fr': 'Janvier',
##                  'ro': 'Ianuarie',
##                  'it': 'Gennaio',
##                  'ja': '1',
##                  'no': 'Januar',
##                  'pl': 'Stycze',
##                  'pt': 'Janeiro',
##                  'zh_TW': '@',
##                  'ru': ''},
##    'February' : {'ca': 'Febrer',
##                  'da': 'Februar',
##                  'de': 'Februar',
##                  'du': 'Februari',
##                  'es': 'Febrero',
##                  'fi': 'Helmikuu',
##                  'fr': 'F&eacute;vrier',
##                  'ro': 'Februarie',
##                  'it': 'Febbraio',
##                  'ja': '2',
##                  'no': 'Februar',
##                  'pl': 'Luty',
##                  'pt': 'Fevereiro',
##                  'zh_TW': 'G',
##                  'ru': ''},
##    'March'    : {'ca': 'Mar&ccedil;',
##                  'da': 'Marts',
##                  'de': 'M&auml;rz',
##                  'du': 'Maart',
##                  'es': 'Marzo',
##                  'fi': 'Maaliskuu',
##                  'fr': 'Mars',
##                  'ro': 'Martie',
##                  'it': 'Marzo',
##                  'ja': '3',
##                  'no': 'Mars',
##                  'pl': 'Marzec',
##                  'pt': 'Maro',
##                  'zh_TW': 'T',
##                  'ru': ''},
##    'April'    : {'ca': 'Abril',
##                  'da': 'April',
##                  'de': 'April',
##                  'du': 'April',
##                  'es': 'Abril',
##                  'fi': 'Huhtikuu',
##                  'fr': 'Avril',
##                  'ro': 'Aprilie',
##                  'it': 'Aprile',
##                  'ja': '4',
##                  'no': 'April',
##                  'pl': 'Kwiecie',
##                  'pt': 'Abril',
##                  'zh_TW': '|',
##                  'ru': ''},
##    'May'      : {'ca': 'Maig',
##                  'da': 'Maj',
##                  'de': 'Mai',
##                  'du': 'Mei',
##                  'es': 'Mayo',
##                  'fi': 'Toukokuu',
##                  'fr': 'Mai',
##                  'ro': 'Mai',
##                  'it': 'Maggio',
##                  'ja': '5',
##                  'no': 'Mai',
##                  'pl': 'Maj',
##                  'pt': 'Maio',
##                  'zh_TW': '',
##                  'ru': ''},
##    'June'     : {'ca': 'Juny',
##                  'da': 'Juni',
##                  'de': 'Juni',
##                  'du': 'Juni',
##                  'es': 'Junio',
##                  'fi': 'Keskuu',
##                  'fr': 'Juin',
##                  'ro': 'Iunie',
##                  'it': 'Giugno',
##                  'ja': '6',
##                  'no': 'Juni',
##                  'pl': 'Czerwiec',
##                  'pt': 'Junho',
##                  'zh_TW': '',
##                  'ru': ''},
##    'July'     : {'ca': 'Juliol',
##                  'da': 'Juli',
##                  'de': 'Juli',
##                  'du': 'Juli',
##                  'es': 'Julio',
##                  'fi': 'Heinkuu',
##                  'fr': 'Juillet',
##                  'ro': 'Iulie',
##                  'it': 'Luglio',
##                  'ja': '7',
##                  'no': 'Juli',
##                  'pl': 'Lipiec',
##                  'pt': 'Julho',
##                  'zh_TW': 'C',
##                  'ru': ''},
##    'August'   : {'ca': 'Agost',
##                  'da': 'August',
##                  'de': 'August',
##                  'du': 'Augustus',
##                  'es': 'Agosto',
##                  'fi': 'Elokuu',
##                  'fr': 'Ao&ucirc;t',
##                  'ro': 'August',
##                  'it': 'Agosto',
##                  'ja': '8',
##                  'no': 'August',
##                  'pl': 'Sierpie',
##                  'pt': 'Agosto',
##                  'zh_TW': 'K',
##                  'ru': ''},
##    'September': {'ca': 'Setembre',
##                  'da': 'September',
##                  'de': 'September',
##                  'du': 'September',
##                  'es': 'Septiembre',
##                  'fi': 'Syyskuu',
##                  'fr': 'Septembre',
##                  'ro': 'Septembrie',
##                  'it': 'Settembre',
##                  'ja': '9',
##                  'no': 'September',
##                  'pl': 'Wrzesie',
##                  'pt': 'Setembro',
##                  'zh_TW': 'E',
##                  'ru': ''},
##    'October'  : {'ca': 'Octubre',
##                  'da': 'Oktober',
##                  'de': 'Oktober',
##                  'du': 'October',
##                  'es': 'Octubre',
##                  'fi': 'Lokakuu',
##                  'fr': 'Octobre',
##                  'ro': 'Octombrie',
##                  'it': 'Ottobre',
##                  'ja': '10',
##                  'no': 'Oktober',
##                  'pl': 'Padziernik',
##                  'pt': 'Outubro',
##                  'zh_TW': 'Q',
##                  'ru': ''},
##    'November' : {'ca': 'Novembre',
##                  'da': 'November',
##                  'de': 'November',
##                  'du': 'November',
##                  'es': 'Noviembre',
##                  'fi': 'Marraskuu',
##                  'fr': 'Novembre',
##                  'ro': 'Noiembrie',
##                  'it': 'Novembre',
##                  'ja': '11',
##                  'no': 'November',
##                  'pl': 'Listopad',
##                  'pt': 'Novembro',
##                  'zh_TW': 'Q@',
##                  'ru': ''},
##    'December' : {'ca': 'Desembre',
##                  'da': 'December',
##                  'de': 'Dezember',
##                  'du': 'December',
##                  'es': 'Diciembre',
##                  'fi': 'Joulukuu',
##                  'fr': 'D&eacute;cembre',
##                  'ro': 'Decembrie',
##                  'it': 'Dicembre',
##                  'ja': '12',
##                  'no': 'Desember',
##                  'pl': 'Grudzie',
##                  'pt': 'Dezembro',
##                  'zh_TW': 'QG',
##                  'ru': ''},
##    'Jan': {'ca': 'Gen',
##            'da': 'Jan',
##            'de': 'Jan',
##            'du': 'Jan',
##            'es': 'Ene',
##            'fi': 'Tam',
##            'fr': 'Jan',
##            'ro': 'Ian',
##            'it': 'Gen',
##            'ja': '1',
##            'no': 'Jan',
##            'pl': 'Sty',
##            'pt': 'Jan',
##            'zh_TW': '@',
##            'ru': ''},
##    'Feb': {'ca': 'Feb',
##            'da': 'Feb',
##            'de': 'Feb',
##            'du': 'Feb',
##            'es': 'Feb',
##            'fi': 'Hel',
##            'fr': 'F&eacute;v',
##            'ro': 'Feb',
##            'it': 'Feb',
##            'ja': '2',
##            'no': 'Feb',
##            'pl': 'Lut',
##            'pt': 'Fev',
##            'zh_TW': 'G',
##            'ru': ''},
##    'Mar': {'ca': 'Mar',
##            'da': 'Mar',
##            'de': 'M&auml;r',
##            'du': 'Maa',
##            'es': 'Mar',
##            'fi': 'Maa',
##            'fr': 'Mar',
##            'ro': 'Mar',
##            'it': 'Mar',
##            'ja': '3',
##            'no': 'Mar',
##            'pl': 'Mar',
##            'pt': 'Mar',
##            'zh_TW': 'T',
##            'ru': ''},
##    'Apr': {'ca': 'Abr',
##            'da': 'Apr',
##            'de': 'Apr',
##            'du': 'Apr',
##            'es': 'Abr',
##            'fi': 'Huh',
##            'fr': 'Avr',
##            'ro': 'Apr',
##            'it': 'Apr',
##            'ja': '4',
##            'no': 'Apr',
##            'pl': 'Kwi',
##            'pt': 'Abr',
##            'zh_TW': '|',
##            'ru': ''},
##    'May': {'ca': 'Mai',
##            'da': 'Maj',
##            'de': 'Mai',
##            'du': 'Mei',
##            'es': 'May',
##            'fi': 'Tou',
##            'fr': 'Mai',
##            'ro': 'Mai',
##            'it': 'Mag',
##            'ja': '5',
##            'no': 'Mai',
##            'pl': 'Maj',
##            'pt': 'Mai',
##            'zh_TW': '',
##            'ru': ''},
##    'Jun': {'ca': 'Jun',
##            'da': 'Jun',
##            'de': 'Jun',
##            'du': 'Jun',
##            'es': 'Jun',
##            'fi': 'Kes',
##            'fr': 'Juin',
##            'ro': 'Iun',
##            'it': 'Giu',
##            'ja': '6',
##            'no': 'Jun',
##            'pl': 'Cze',
##            'pt': 'Jun',
##            'zh_TW': '',
##            'ru': ''},
##    'Jul': {'ca': 'Jul',
##            'da': 'Jul',
##            'de': 'Jul',
##            'du': 'Jul',
##            'es': 'Jul',
##            'fi': 'Hei',
##            'fr': 'Juil',
##            'ro': 'Iul',
##            'it': 'Lug',
##            'ja': '7',
##            'no': 'Jul',
##            'pl': 'Lip',
##            'pt': 'Jul',
##            'zh_TW': 'C',
##            'ru': ''},
##    'Aug': {'ca': 'Ago',
##            'da': 'Aug',
##            'de': 'Aug',
##            'du': 'Aug',
##            'es': 'Ago',
##            'fi': 'Elo',
##            'fr': 'Ao&ucirc;',
##            'ro': 'Aug',
##            'it': 'Ago',
##            'ja': '8',
##            'no': 'Aug',
##            'pl': 'Sie',
##            'pt': 'Ago',
##            'zh_TW': 'K',
##            'ru': ''},
##    'Sep': {'ca': 'Set',
##            'da': 'Sep',
##            'de': 'Sep',
##            'du': 'Sep',
##            'es': 'Sep',
##            'fi': 'Syy',
##            'fr': 'Sep',
##            'ro': 'Sep',
##            'it': 'Set',
##            'ja': '9',
##            'no': 'Sep',
##            'pl': 'Wrz',
##            'pt': 'Set',
##            'zh_TW': 'E',
##            'ru': ''},
##    'Oct': {'ca': 'Oct',
##            'da': 'Okt',
##            'de': 'Okt',
##            'du': 'Oct',
##            'es': 'Oct',
##            'fi': 'Lok',
##            'fr': 'Oct',
##            'ro': 'Oct',
##            'it': 'Ott',
##            'ja': '10',
##            'no': 'Okt',
##            'pl': 'Pa',
##            'pt': 'Out',
##            'zh_TW': 'Q',
##            'ru': ''},
##    'Nov': {'ca': 'Nov',
##            'da': 'Nov',
##            'de': 'Nov',
##            'du': 'Nov',
##            'es': 'Nov',
##            'fi': 'Mar',
##            'fr': 'Nov',
##            'ro': 'Nov',
##            'it': 'Nov',
##            'ja': '11',
##            'no': 'Nov',
##            'pl': 'Lis',
##            'pt': 'Nov',
##            'zh_TW': 'Q@',
##            'ru': ''},
##    'Dec': {'ca': 'Des',
##            'da': 'Dec',
##            'de': 'Dez',
##            'du': 'Dec',
##            'es': 'Dic',
##            'fi': 'Jou',
##            'fr': 'D&eacute;c',
##            'ro': 'Dec',
##            'it': 'Dec',
##            'ja': '12',
##            'no': 'Des',
##            'pl': 'Gru',
##            'pt': 'Dez',
##            'zh_TW': 'QG',
##            'ru': ''},
##    'Sunday'   : {'ca': 'Diumenge',
##                  'da': 'Sndag',
##                  'de': 'Sonntag',
##                  'du': 'Zondag',
##                  'es': 'Domingo',
##                  'fi': 'Sunnuntai',
##                  'fr': 'Dimanche',
##                  'ro': 'Duminica',
##                  'it': '',
##                  'no': 'S&oslash;ndag',
##                  'pl': 'Niedziela',
##                  'pt': 'Domingo',
##                  'ru': ''},
##    'Monday'   : {'ca': 'Dilluns',
##                  'da': 'Mandag',
##                  'de': 'Montag',
##                  'du': 'Maandag',
##                  'es': 'Lunes',
##                  'fi': 'Maanantai',
##                  'fr': 'Lundi',
##                  'ro': 'Luni',
##                  'it': '',
##                  'no': 'Mandag',
##                  'pl': 'Poniedziaek',
##                  'pt': 'Segunda',
##                  'ru': ''},
##    'Tuesday'  : {'ca': 'Dimarts',
##                  'da': 'Tirsdag',
##                  'de': 'Dienstag',
##                  'du': 'Dinsdag',
##                  'es': 'Martes',
##                  'fi': 'Tiistai',
##                  'fr': 'Mardi',
##                  'ro': 'Marti',
##                  'it': '',
##                  'no': 'Tirsdag',
##                  'pl': 'Wtorek',
##                  'pt': 'Tera',
##                  'ru': ''},
##    'Wednesday': {'ca': 'Dimecres',
##                  'da': 'Onsdag', 
##                  'de': 'Mittwoch',
##                  'du': 'Woensdag',
##                  'es': 'Mi&eacute;rcoles',
##                  'fi': 'Keskiviikko',
##                  'fr': 'Mercredi',
##                  'ro': 'Miercuri',
##                  'it': '',
##                  'no': 'Onsdag',
##                  'pl': 'roda',
##                  'pt': 'Quarta',
##                  'ru': ''},
##    'Thursday' : {'ca': 'Dijous',
##                  'da': 'Torsdag',
##                  'de': 'Donnerstag',
##                  'du': 'Donderdag',
##                  'es': 'Jueves',
##                  'fi': 'Torstai',
##                  'fr': 'Jeudi',
##                  'ro': 'Joi',
##                  'it': '',
##                  'no': 'Torsdag',
##                  'pl': 'Czwartek',
##                  'pt': 'Quinta',
##                  'ru': ''},
##    'Friday'   : {'ca': 'Divendres',
##                  'da': 'Fredag',
##                  'de': 'Freitag',
##                  'du': 'Vrijdag',
##                  'es': 'Viernes',
##                  'fi': 'Perjantai',
##                  'fr': 'Vendredi',
##                  'ro': 'Vineri',
##                  'it': '',
##                  'no': 'Fredag',
##                  'pl': 'Pitek',
##                  'pt': 'Sexta',
##                  'ru': ''},
##    'Saturday' : {'ca': 'Dissabte',
##                  'da': 'L&slash;rdag',
##                  'de': 'Samstag',
##                  'du': 'Zaterdag',
##                  'es': 'S&aacute;bado',
##                  'fi': 'Lauantai',
##                  'fr': 'Samedi',
##                  'ro': 'S&icirc;mbata',
##                  'it': '',
##                  'no': 'L&oslash;rdag',
##                  'pl': 'Sobota',
##                  'pt': 'Sbado',
##                  'ru': ''}
                }

def translate(word, lang):
    try:
        return translations[word][lang]
    except KeyError:
        return word


def datestr(date):
    # note we must use yy/mm/dd instead of yy-mm-dd
    # to preserve the date time in localtime
    return '%d/%s/%s' % (date.year(), date.mm(), date.dd())


def name_or_expr(mapping, name_attr, expr_attr, default):
    name = mapping.get(name_attr, None)
    expr = mapping.get(expr_attr, None)

    if name is None:
        if expr is None:
            return default
        return Eval(expr)
    if expr is None:
        return name
    raise ParseError, ('%s and %s given' % (name_attr, expr_attr), 'calendar')

def get_value(x, md):
    if type(x) is StringType:
        return x
    else:
        return x.eval(md)


class CalendarTag:
    """
    Calendar tag.
    
    See README.txt for usage, author, copyright, and license.
    """

    name = 'calendar'
    blockContinuations = ()
    expr=None

    def __init__(self, blocks):
        tname, args, section = blocks[0]
        self.section = section.blocks
        self.args = parse_params(args, name='', expr='',
                                 bgcolor='', bgcolor_expr='', tablewidth='',
                                 mode='', date='',
                                 controls='', weekdays='',
                                 lang='', lang_expr='',
                                 modes='', images='', theme='')

        self.__name__ = name_or_expr(self.args, 'name', 'expr', 'calendar')

        modes = self.args.get('modes', None)
        if modes is None:
            self.modes = ('day', 'week', 'month', 'year')
        else:
            self.modes = split(modes, ',')
            for mode in self.modes:
                if mode not in ('day', 'week', 'month', 'year'):
                    raise "Unallowed mode: %s" % mode

        self.theme = self.args.get('theme', '')
        if not self.theme in ('','yb'):
            raise "Unallowed theme %s" % self.theme
 
        self.__bgcolor__ = name_or_expr(self.args, 'bgcolor', 'bgcolor_expr',
                                        '#ffffff')

        self.tablewidth = self.args.get('tablewidth', None)

        self.defaultMode = self.args.get('mode', 'month')
        if self.defaultMode not in self.modes:
            if 'month' in self.modes:
                self.defaultMode = 'month'
            else:
                self.defaultMode = self.modes[0]
        self.date = self.args.get('date', None)

        self.controls = self.args.get('controls', 'yes')

        self.images = self.args.get('images','yes')

        weekdays = self.args.get('weekdays', '07')
        self.week_sday = int(weekdays[0])
        self.week_ndays = int(weekdays[1])

        self.__lang__ = name_or_expr(self.args, 'lang', 'lang_expr', 'en')

    def render(self, md):
##        time_start = time()

        name = get_value(self.__name__, md)
        lang = get_value(self.__lang__, md)
        self.bgcolor = get_value(self.__bgcolor__, md)

        if md.has_key('date-' + name) and self.controls != 'no':
            date = DateTime(md['date-' + name])
        else:
            if self.date:
                d = Eval(self.date)
                d = d.eval(md)
                if type(d) in (type(0), type('')):
                    date = DateTime(d)
                else:
                    date = DateTime(str(d))
            else:
                date = DateTime()

        date = date.earliestTime() + 0.5

        if md.has_key('mode-' + name) and self.controls != 'no':
            mode = md['mode-' + name]
        else:
            mode = self.defaultMode

        if mode not in self.modes:
            raise "Unallowed mode: %s" % mode

        self.vals_ = v = {}

        ## Translations
        for word in translations.keys():
            v[word] = translate(word, lang)


        v['name_'] = name
        v['date_'] = datestr(date)
        v['date'] = date
        v['mode'] = mode
        v['start_optional']=""

        v['spacing'] = '0'
        v['padding'] = '4'
        v['bordercolor'] = '#000000'

        v['bgcolor'] = v['leftfgcolor'] = v['middlebgcolor'] = \
            v['rightbgcolor'] = v['daybgcolor'] = v['mhbgcolor'] = self.bgcolor
        v['leftbgcolor'] = '#000000'

        v['ipath_'] = md['BASE1'] + '/misc_/Calendar/'

        l = ['mode-' + name, 'date-' + name]
        params = filter(lambda x: x, split(md['QUERY_STRING'], '&'))
        params = map(lambda x: split(x, '='), params)
        params = filter(lambda x, l=l: x[0] not in l, params)
        params = map(lambda x: join(x, '='), params)

        v['url_'] = md['URL'] + '?' + join(params, '&amp;')

        self.__setupImageUrls(v, mode)

        v['day_text_'] = v['Day']
        v['week_text_'] = v['Week']
        v['month_text_'] = v['Month']
        v['year_text_'] = v['Year']

        v['rightmiddle'] = '&nbsp;'

        v['prev_alt_'] = v['Previous Week']
        v['prev_url_'] = self.linkDate_(date - 7, mode)

        v['next_alt_'] = v['Next Week']
        v['next_url_'] = self.linkDate_(date + 7, mode)

        try:
            v['daynames'] = daynames[lang]
        except KeyError:
            v['daynames'] = daynames['en']
        try:
            v['monthnames'] = monthnames[lang]
        except KeyError:
            v['monthnames'] = monthnames['en']
        try:
            v['shortmonthnames'] = shortmonthnames[lang]
        except KeyError:
            v['shortmonthnames'] = shortmonthnames['en']

        ##
        if self.controls == 'no':
            v['colspan'] = 1
        else:
            v['colspan'] = 3


        if self.tablewidth is not None:
            v['tablewidth'] = ' width="%s"' % self.tablewidth
        else:
            v['tablewidth'] = ''

        ## Calculate the body
        if mode == 'year':
            s = '<tr>'
            d = date - (date.dayOfYear() - 1)
            for i in range(1,13):
                body = self.render_dwm('month', md, d)
                s = s + '<td valign="top"><table cellpadding="0" cellspacing="4" border="0">'
                s = s + (body_start % v) +\
                    '<tr><td bgcolor="%s" colspan="%d" style="font-size:9px;">'\
                    % (v['mhbgcolor'], self.week_ndays)
                if self.controls != 'no':
                    s = s + '<a href="%s">' % self.linkDate_(d, 'month')
                s = s + v['monthnames'][d.month()-1]
                if self.controls != 'no':
                    s = s + '</a>'
                s = s + '</td></tr>' + body + body_end + '</table></td>'
                if i%3 == 0:
                    s = s + '</tr><tr><td>&nbsp;</td></tr><tr>'
                else:
                    s = s + '<td> &nbsp; </td>'
                d = d + 40
                d = d - (d.day() - 1)
            s = s + '</tr>'
            v['start_optional']="""
</table>
<table border="0" bgcolor="%(bgcolor)s" cellspacing="0" cellpadding="0"%(tablewidth)s>
            """ % v
        else:
            body = self.render_dwm(mode, md, date)
            s = (body_start % v) + body + body_end

        ## Calculate left
	v['Y']=str(date.year())
	v['y']=date.yy()
	v['M']=v['monthnames'][date.month()-1]
	v['S']=v['shortmonthnames'][date.month()-1]
	v['m']=v['monthnames'][date.month()-1].lower()
	v['s']=v['shortmonthnames'][date.month()-1].lower()
	v['h']=date.mm()
	v['d']=date.dd()
        v['w']=str(((date - ((date.dow()+(7-self.week_sday)) % 7)).dayOfYear() / 7) + 1)
        if mode == 'year':
            if v.has_key('left'):
		left=v['left'].replace('yyyy','%(Y)s').replace('yy','%(y)s').replace('Mmmm','%(M)s').replace('Mmm','%(S)s').replace('mmmm','%(m)s').replace('mmm','%(s)s').replace('mm','%(h)s').replace('dd','%(d)s')  % v
	    else:
		left = str(date.year())
            v['year_text_'] = '<b>%(year_text_)s</b>' % v
            v['prev_alt_'] = v['Previous Year']
            v['next_alt_'] = v['Next Year']

            d = date - date.dayOfYear()
            v['prev_url_'] = self.linkDate_(d, mode)
            if date.isLeapYear():
                d = date + 366
            else:
                d = date + 365
            v['next_url_'] = self.linkDate_(d, mode)
        elif mode == 'day':
            monthname = v['shortmonthnames'][date.month()-1]
            if v.has_key('left'):
		left=v['left'].replace('yyyy','%(Y)s').replace('yy','%(y)s').replace('Mmmm','%(M)s').replace('Mmm','%(S)s').replace('mmmm','%(m)s').replace('mmm','%(s)s').replace('mm','%(h)s').replace('dd','%(d)s')  % v
            elif v.has_key('month_day_year_format'):
		left=v['month_day_year_format'].replace('yyyy','%(Y)s').replace('yy','%(y)s').replace('Mmmm','%(M)s').replace('Mmm','%(S)s').replace('mmmm','%(m)s').replace('mmm','%(s)s').replace('mm','%(h)s').replace('dd','%(d)s')  % v
	    else:
		left = '%s %d, %d' % (monthname, date.day(), date.year())
        elif mode == 'week':
            sdow = date - ((date.dow()+(7-self.week_sday)) % 7)
            edow = sdow + self.week_ndays -1
            smonthname = v['shortmonthnames'][sdow.month()-1]
            emonthname = v['shortmonthnames'][edow.month()-1]
	    v['StY']=str(sdow.year())
	    v['Sty']=sdow.yy()
	    v['StM']=v['monthnames'][sdow.month()-1]
	    v['StS']=v['shortmonthnames'][sdow.month()-1]
	    v['Stm']=v['monthnames'][sdow.month()-1].lower()
	    v['Sts']=v['shortmonthnames'][sdow.month()-1].lower()
	    v['Sth']=sdow.mm()
	    v['EnY']=str(edow.year())
	    v['Eny']=edow.yy()
	    v['EnM']=v['monthnames'][edow.month()-1]
	    v['EnS']=v['shortmonthnames'][edow.month()-1]
	    v['Enm']=v['monthnames'][edow.month()-1].lower()
	    v['Ens']=v['shortmonthnames'][edow.month()-1].lower()
	    v['Enh']=edow.mm()

            if v.has_key('left'):
		left=v['left'].replace('yyyy','%(Y)s').replace('yy','%(y)s').replace('Mmmm','%(M)s').replace('Mmm','%(S)s').replace('mmmm','%(m)s').replace('mmm','%(s)s').replace('mm','%(h)s').replace('dd','%(d)s')  % v
            elif v.has_key('shortmonth_year_format'):
		left=v['shortmonth_year_format'].replace('yyyy','%(Y)s').replace('yy','%(y)s').replace('Mmmm','%(M)s').replace('Mmm','%(S)s').replace('mmmm','%(m)s').replace('mmm','%(s)s').replace('mm','%(h)s').replace('dd','%(d)s')
		if sdow.year() != edow.year():
		    left=left.replace('%(M)s','%(StM)s').replace('%(S)s','%(StS)s').replace('%(m)s','%(Stm)s').replace('%(s)s','%(Sts)s').replace('%(h)s','%(Sth)s').replace('%(Y)s','%(StY)s').replace('%(y)s','%(Sty)s')+' - '+left.replace('%(M)s','%(EnM)s').replace('%(S)s','%(EnS)s').replace('%(m)s','%(Enm)s').replace('%(s)s','%(Ens)s').replace('%(h)s','%(Enh)s').replace('%(Y)s','%(EnY)s').replace('%(y)s','%(Eny)s')
		elif sdow.month() != edow.month():
		    left=left.replace('%(M)s','%(StM)s-%(EnM)s').replace('%(S)s','%(StS)s-%(EnS)s').replace('%(m)s','%(Stm)s-%(Enm)s').replace('%(s)s','%(Sts)s-%(Ens)s').replace('%(h)s','%(Sth)s-%(Enh)s')
		left=left % v
	    else:
                left = '%s %d' % (emonthname, edow.year())
                if sdow.aMonth() != edow.aMonth():
                    left = ' - ' + left
                    if sdow.year() != edow.year():
                        left = (' %d' % sdow.year()) + left
		    left = smonthname + left

            v['week_text_'] = '<b>%(week_text_)s</b>' % v
        else:
            if v.has_key('left'):
		left=v['left'].replace('yyyy','%(Y)s').replace('yy','%(y)s').replace('Mmmm','%(M)s').replace('Mmm','%(S)s').replace('mmmm','%(m)s').replace('mmm','%(s)s').replace('mm','%(h)s').replace('dd','%(d)s')  % v
            elif v.has_key('longmonth_year_format'):
		left=v['longmonth_year_format'].replace('yyyy','%(Y)s').replace('yy','%(y)s').replace('Mmmm','%(M)s').replace('Mmm','%(S)s').replace('mmmm','%(m)s').replace('mmm','%(s)s').replace('mm','%(h)s').replace('dd','%(d)s')  % v
	    else:
                left = '%s %d' % (v['monthnames'][date.month()-1], date.year())
            v['month_text_'] = '<b>%(month_text_)s</b>' % v
            v['prev_alt_'] = v['Previous Month']
            v['next_alt_'] = v['Next Month']

            d = date - date.day()
            v['prev_url_'] = self.linkDate_(d, mode)

            d = date - (date.day() - 40)
            v['next_url_'] = self.linkDate_(d, mode)

        if self.images != 'no':
            v['prev_link_'] = '<img align="middle" width="8" height="17" border="0" alt="%(prev_alt_)s" src="%(ipath_)slarrow">'% v
            v['next_link_'] = '<img align="middle" width="8" height="17" border="0" alt="%(next_alt_)s" src="%(ipath_)srarrow">'% v
        else:
            v['prev_link_'] = '&nbsp;' + v['prev_alt_'] + '&nbsp;'
            v['next_link_'] = '&nbsp;' + v['next_alt_'] + '&nbsp;'

        v['left'] = '&nbsp;&nbsp;' + left + '&nbsp;&nbsp;'

        ## Calculate middle and right
        if self.controls == 'no':
            v['middle'] = ''
            v['right'] = ''
        else:
            if mode == 'day':
                rmid = ''
                if date.dow() >= self.week_sday:
                    d = date - date.dow()
                else:
                    d = date - 7
                d = d + self.week_sday
                for i in range(self.week_ndays):
                    weekday = d.dow()
                    if date.dow() == weekday:
                        color = '#ffffff'
                        bgcolor = '#000000'
                    else:
                        color = '#000000'
                        bgcolor = '#ffffff'

                    v['dow_date_'] = datestr(d)
                    dow_url = "%(url_)s&amp;date-%(name_)s=%(dow_date_)s&amp;mode-%(name_)s=%(mode)s" % v

                    rmid = rmid + '''<a class="caldate" style="background-color:%s;color:%s" href="%s">%s</a>''' % \
                                (bgcolor, color, dow_url, v['daynames'][weekday])
                    d = d + 1

                v['rightmiddle'] = rmid 
            elif mode == 'week':
		if v.has_key('week_format'):
		    v['rightmiddle']='&nbsp;'+v['week_format']+'&nbsp;'
		    v['rightmiddle']=v['rightmiddle'].replace('ww','%(w)s') % v
		else:
		    v['rightmiddle'] = '&nbsp;%s %s&nbsp;' %\
                                   (v['Week'] , v['w'])

            if not v.has_key('middle'):
                middle = '<td align="center" valign="middle" bgcolor="%(middlebgcolor)s"><table border=0 cellspacing=0 cellpadding=0><tr><td>&nbsp;</td>'
                if self.images != 'no':
                    for mode in self.modes:
                        middle = middle + '''<td>
    <a href="%s&amp;date-%s=%s&amp;mode-%s=%s"><img border="0" src="%s%s" alt="%s"></a></td>''' % (replace(v['url_'], '%', '%%'), v['name_'], v['date_'], v['name_'], mode, v['ipath_'], v[mode+'_img_'], v[capitalize(mode)])
                else:
                    for mode in self.modes:
                        middle = middle + '<td>&nbsp;<a href="%s&amp;date-%s=%s&amp;mode-%s=%s">%s</a>&nbsp;</td>' % (replace(v['url_'], '%', '%%'), v['name_'], v['date_'], v['name_'], mode, v[mode+'_text_'])
                v['middle'] = middle + "</tr></table></td>"
            v['middle'] = v['middle'] % v

            if not v.has_key('right'):
                v['right'] = right
            v['right'] = v['right'] % v


##        time_end = time()
##        time_diff = time_end - time_start

        return (start % v) + s + end 

    __call__ = render


    def render_dwm(self, mode, md, date):
        v = self.vals_

        if mode == 'day':
            s = '<tr>%s</tr>' % self.render_day_(md, date, 1, mode)
        else:
            s = '<tr>'
            if mode == 'week':
                while date.dow() != self.week_sday:
                    date = date - 1
                for i in range(self.week_ndays):
                    s = s + self.render_day_(md, date + i, i == 0, mode)
            elif mode == 'month':
                bd = date - (date.day() - 1)
                c = 0
                while (bd - c).dow() != self.week_sday:
                    c = c + 1
                if c > 0:
                    s = s + '<td bgcolor="%s" colspan="%s">&nbsp;</td>' % (
                        self.bgcolor, c)

                d = bd
                month = bd.month()
                while d.month() == month:
                    if c == 7:
                        s = s + '</tr><tr>'
                        c = 0
                    if c < self.week_ndays:
                        s = s + self.render_day_(md, d, d == bd, mode)
                    d = d + 1
                    c = c + 1
                if c < self.week_ndays:
                    s = s + '<td bgcolor="%s" colspan="%s">&nbsp;</td>' % (
                        self.bgcolor, self.week_ndays - c)
            s = s + '</tr>'

            if v['daynames']:
                x = ''
                c = v['daybgcolor']

                for d in range(self.week_sday, self.week_ndays+self.week_sday):
                    x = x + '<td bgcolor="%s" width="%d%%" style="font-size:9px">%s</td>' \
                        % (c, 100/self.week_ndays, v['daynames'][d%7])

                s = '<tr>%s</tr>' % x + s

        return s


    def render_day_(self, md, date, firstday, mode):
        date = date.earliestTime()

        ns = namespace(md,
            date = date,
            mode = mode,
            firstday = firstday,
            setCalendar = self.setCalendar_,
            getCalendar = self.getCalendar_,
            linkDate = self.linkDate_
        )[0]

        md._push(InstanceDict(ns, md))

        v = self.vals_
        v['cellbgcolor'] = self.bgcolor
        v['align'] = None
        v['valign'] = None

        v['cellattrs'] = None

        try:
            s = render_blocks(self.section, md)
        finally:
            md._pop(1)
            pass

        x = v['cellattrs']
        if x is None:
            pre = '<td bgcolor="%s" class="caldate"' % v['cellbgcolor']
            x = v['align']
            if x is not None:
                pre = pre + ' align="%s"' % x
            x = v['valign']
            if x is not None:
                pre = pre + ' valign="%s"' % x
        else:
            pre = '<td ' + join(map(lambda x: '%s="%s"' % (x[0],x[1]), x))
        return pre + '>' + s + '</td>\n'


    def linkDate_(self, date, mode = 'day'):
        if mode not in self.modes:
            mode = 'day'

        v = self.vals_
        s = '%(url_)s&amp;date-%(name_)s=' % v
        return s + datestr(date) + ('&amp;mode-%(name_)s=' % v) + mode

        
    def setCalendar_(self, name, val):
        self.vals_[name] = val


    def getCalendar_(self, name, defval = None):
        if self.vals_.has_key(name):
            return self.vals_[name]
        else:
            return defval
	
    def __setupImageUrls(self, v, mode=''):
        # image urls
        if self.theme: theme = self.theme + "_" 
        else: theme = ""
        v['day_img_']=theme + 'day'
        v['week_img_']=theme + 'week'
        v['month_img_']=theme + 'month'
        v['year_img_']=theme + 'year'
        if mode == 'day':
            v['day_img_']=theme + 'sday'
        elif mode == 'week':
            v['week_img_']=theme + 'sweek'
        elif mode == 'month':
            v['month_img_']=theme + 'smonth'
        elif mode == 'year':
            v['year_img_']=theme + 'syear'

# Install the calendar tag globally

String.commands['calendar'] = CalendarTag
