#
# Copyright 2001 Stfane Fermigier and Nuxeo SARL
# See LICENSE.TXT for licensing information
#

"""NuxDocument: a Zope product that represents generic documents by using
plugins to convert native productivity suite formats to HTML (for viewing or
previewing in a regular browser) or to plain text (for indexing).
"""

__version__='$Revision: 1.34 $'[11:-2]

from OFS.SimpleItem import SimpleItem, Item_w__name__
from OFS.ObjectManager import ObjectManager
from OFS.Cache import Cacheable
from OFS.History import Historical
from OFS.Image import File
import Acquisition
from DateTime.DateTime import DateTime

from AccessControl import ClassSecurityInfo
from Globals import InitializeClass

from urllib import quote
import re, string, traceback

import plugins, config


class NuxDocument(SimpleItem, ObjectManager):
  # (SimpleItem):
  # (CatalogAware, SimpleItem, Acquisition.Implicit, 
  # PropertyManager, Historical, Cacheable, Item_w__name__):
  # + ElementWithTitle (XXX ???)

  """The NuxDocument base class. Provides only state and behavior,
  UI in provided by subclasses."""

  index_html = None # Prevent accidental acquisition (XXX ???)

  filename = _text = _html = ''
  _file = None

  _mime_type = ''

  security = ClassSecurityInfo()

  #
  # Constructor
  #
  def __init__(self, id='', title='', file='', mime_type=''):
    self._setId(id)
    self._edit(title, file, mime_type)


  # taken from CMFDefault.File
  def _isNotEmpty(self, file):
      """Do various checks on 'file' to try to determine non emptiness."""
      if not file:
          return 0                    # Catches None, Missing.Value, ''
      elif file and (type(file) is type('')):
          return 1
      elif getattr(file, 'filename', None):
          return 1
      elif not hasattr(file, 'read'):
          return 0
      else:
          file.seek(0, 2)             # 0 bytes back from end of file
          t = file.tell()             # Report the location
          file.seek(0)                # and return pointer back to 0
          if t: return 1
          else: return 0

  def _edit(self, title=None, file=None, mime_type=None):
    filename = getattr(file, 'filename', self.__name__)
    self.filename = filename
    if title is not None:
      self.title = title
    notempty = self._isNotEmpty(file)
    if notempty:
      self._file = File(self.__name__, title, file)
    # set a default mime_type that convert() can override
    user_mime_type = ''
    if mime_type:
      user_mime_type = mime_type
      self._mime_type = mime_type
    elif notempty:
      mime_type = self._file.content_type
      if mime_type not in ('',
                           'application/octet-stream',
                           'application/unknown'):
        user_mime_type = mime_type
      self._mime_type = mime_type
    if notempty:
      self.convert()
      # if user/browser set the mime type, keep it
      if user_mime_type:
        self._mime_type = user_mime_type
    self.publication_date = DateTime()

  def convert(self):
    data = self.getRaw()
    plugin = plugins.selectPlugin(data, mime_type=self.MimeType(),
                                  filename=self.filename)

    if plugin != plugins.Dumb:
      try:
        result = plugin.getConverter(data)
      except plugins.ConversionError:
        #traceback.print_exc()
        plugin = plugins.getDefaultPlugin()
        result = plugin.getConverter(data)
    else:
      result = plugin.getConverter(data)

    self._mime_type = result.getMimeType() or self.MimeType()
    self._text = result.getText()
    self._html = result.getBody()
    if not self.title:
      self.title = result.getTitle()
    for image_name in result.getImageNames():
      try:
        self.manage_addImage(image_name, result.getImage(image_name))
      except:
        self.manage_delObjects(ids=[image_name])
        self.manage_addImage(image_name, result.getImage(image_name))

  #
  # Accessors / rendering
  #
  security.declarePrivate('getRaw')
  def getRaw(self):
    "Get raw content of document"

    return str(self._file)
  
  security.declareProtected('View', 'getHtml')
  def getHtml(self):
    "Get raw HTML converted content"

    return self._html

  security.declareProtected('View', 'getFileSize')
  def getFileSize(self):
    "Get raw content size in bytes"
    return len(self.getRaw())

  security.declareProtected('View', 'getFileSizeString')
  def getFileSizeString(self):
    """Get raw content size as a dict

    E.g.: { 'size': size, 'unit': 'B'/'KB'/'MB' }, formats size in B/KB/MB
    """
    size = self.getFileSize()
    if size < 1024:
      return { 'size': size, 'unit': 'B' }
    elif size < 1048576:
      return { 'size': size/1024, 'unit': 'KB' }
    else:
      return { 'size': size/1048576, 'unit': 'MB' }

  security.declareProtected('View', 'MimeType')
  def MimeType(self):
    """Get MIME type"""
    return self._mime_type or 'application/octet-stream'

  security.declareProtected('View', 'document_src')
  def document_src(self, REQUEST=None, RESPONSE=None):
    """Unprocessed document source."""
    if RESPONSE is not None:
      RESPONSE.setHeader('Content-Type', self.MimeType())
      if self.filename:
        RESPONSE.setHeader(
          'Content-Disposition', "inline; filename=%s" % self.filename)
    return self.getRaw()

  security.declareProtected('FTP access', 'manage_FTPget')
  def manage_FTPget(self):
    "Get source for FTP download"

    return self.getRaw()

  security.declareProtected('View', 'PrincipiaSearchSource')
  def PrincipiaSearchSource(self):
    return self._text
    #return string.join(re.split(r'[^%s]+' % string.letters, self._text))

  security.declareProtected('View', 'SearchableText')
  def SearchableText(self):
    "Get content suitable for Cataloging"
    return "%s %s %s" % (
      self.title, self.Description(), self.PrincipiaSearchSource())

  security.declareProtected('Edit NuxDocument', 'PUT')
  def PUT(self, REQUEST, RESPONSE):
    "Handle HTTP/FTP PUT requests."
    self.dav__init(REQUEST, RESPONSE)
    file = REQUEST.get('BODY', '')
    self._edit(file=file)
    RESPONSE.setStatus(204)
    return RESPONSE

InitializeClass(NuxDocument)

